#include "transforming_consumer.h"

#include <solomon/agent/misc/labels.h>
#include <solomon/agent/misc/logger.h>

#include <solomon/libs/cpp/glob/glob.h>
#include <solomon/libs/cpp/labels/validate.h>

#include <library/cpp/monlib/metrics/metric_value.h>

#include <util/datetime/base.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/string/escape.h>
#include <util/string/split.h>
#include <util/string/strip.h>


namespace NSolomon::NAgent {
namespace {

using TLabelBackend = TAgentLabel::TStringType;

template<typename TStringType>
class TLabelNameHash {
public:
    size_t operator()(TStringBuf name) const {
        return Hasher_(name);
    }

    size_t operator()(const NMonitoring::TLabelImpl<TStringType>& label) const {
        return Hasher_(TStringBuf{label.Name()});
    }

private:
    THash<TStringBuf> Hasher_;
};

template <typename TStringType>
class TLabelNameEqual {
public:
    bool operator()(TStringBuf name, const NMonitoring::TLabelImpl<TStringType>& label) const {
        return name == TStringBuf{label.Name()};
    }

    bool operator()(const NMonitoring::TLabelImpl<TStringType>& label, TStringBuf name) const {
        return this->operator()(name, label);
    }

    bool operator()(TStringBuf l, TStringBuf r) const {
        return l == r;
    }

    bool operator()(const NMonitoring::TLabelImpl<TStringType>& l, const NMonitoring::TLabelImpl<TStringType>& r) const {
        return l.Name() == r.Name();
    }
};

using TLabelsHashMap = THashMap<TLabelBackend, TLabelBackend, TLabelNameHash<TLabelBackend>, TLabelNameEqual<TLabelBackend>>;

std::optional<TStringBuf> GetLabelValueSafe(TStringBuf) {
    return std::nullopt;
}

template <class... TLabelsContainer>
std::optional<TStringBuf> GetLabelValueSafe(TStringBuf name, const TLabelsHashMap& first, const TLabelsContainer&... rest) {
    if (auto it = first.find(name); it != first.end()) {
        return it->second;
    }

    return GetLabelValueSafe(name, rest...);
}

template <class... TLabelsContainer>
TStringBuf GetLabelValue(TStringBuf name, const TLabelsHashMap& first, const TLabelsContainer&... rest) {
    auto res = GetLabelValueSafe(name, first, rest...);

    if (!res.has_value()) {
        ythrow yexception() << "haven't found a label with the name \"" << name << "\""
                            << " inside metric labels";
    }

    return res.value();
}

using THashLabelsMatcher = std::function<bool(const TLabelsHashMap& labels, const TLabelsHashMap* optionalLabels)>;

THashLabelsMatcher CreateGlobMatcher(TLabels&& pattern) {
    return [pattern](const TLabelsHashMap& labels, const TLabelsHashMap* optionalLabels) {
        for (auto&& patternLabel: pattern) {
            std::optional<TStringBuf> labelValue;

            if (optionalLabels) {
                labelValue = GetLabelValueSafe(patternLabel.Name(), labels, *optionalLabels);
            } else {
                labelValue = GetLabelValueSafe(patternLabel.Name(), labels);
            }

            if (!labelValue.has_value()) {
                return false;
            }

            if (!IsGlobMatch(patternLabel.Value(), labelValue.value())) {
                return false;
            }
        }

        return true;
    };
}

bool IsPrintableAndNotEmpty(TStringBuf str) {
    if (str.empty()) {
        return false;
    }

    for (auto& ch: str) {
        if (!(ch >= 32 && ch <= 126)) {
            return false;
        }
    }

    return true;
}

void ValidateMatchLabelName(TStringBuf name) {
    Y_ENSURE(IsPrintableAndNotEmpty(name), "Label name in a Match is empty or has non-printable characters");
}

void ValidateMatchLabelValue(TStringBuf value) {
    // Labels gathered from different data sources: Prometheus Exporter, etc.
    // A value can hold almost any character, so there's no real pattern to test
    Y_ENSURE(IsPrintableAndNotEmpty(value), "Label value in a Match is empty or has non-printable characters");
}

void ValidateExternalLabelName(TStringBuf name) {
    Y_ENSURE(IsPrintableAndNotEmpty(name), "Label name in a substitution is empty or has non-printable characters");
}

void ValidateReplaceMetaLabelValue(TStringBuf value) {
    Y_ENSURE(IsPrintableAndNotEmpty(value), "Label value in a ReplaceMeta is empty or has non-printable characters");
}

class TLabelValueExpr {
public:
    TLabelValueExpr(TStringBuf expr) {
        BuildFrom(expr);
    }

    TVector<TStringBuf> GetNames() const {
        TVector<TStringBuf> result;

        if (ValueNamePairs_.size() < 2) {
            return result;
        }

        for (size_t i = 1; i < ValueNamePairs_.size(); i += 2) {
            if (ValueNamePairs_[i].has_value()) {
                result.emplace_back(ValueNamePairs_[i].value());
            }
        }

        return result;
    }

    template <class... TLabelsContainer>
                       // Labels by priority, from high to low
    TLabelBackend Eval(const TLabelsContainer&... labels) const {
        if (ValueNamePairs_.size() == 1) {
            // shortcut
            return ValueNamePairs_[0].value();
        }

        if (ValueNamePairs_.size() == 2 && !ValueNamePairs_[0].has_value()) {
            // shortcut
            auto value = GetLabelValue(ValueNamePairs_[1].value(), labels...);

            return TLabelBackend{value.data(), value.size()};
        }

        TStringBuf value;
        TLabelBackend result;

        for (size_t i = 0; i < ValueNamePairs_.size(); ++i) {
            if (!ValueNamePairs_[i].has_value()) {
                continue;
            }

            if (i % 2 == 0) {
                value = ValueNamePairs_[i].value();
            } else {
                value = GetLabelValue(ValueNamePairs_[i].value(), labels...);
            }

            result.append(value.data(), value.size());
        }

        return result;
    }

    TString ToString() const {
        TString r;

        for (size_t i = 0; i < ValueNamePairs_.size(); ++i) {
            if (i % 2 != 0) {
                r.append("{{");
            }

            if (ValueNamePairs_[i].has_value()) {
                r.append(ValueNamePairs_[i].value());
            }

            if (i % 2 != 0) {
                r.append("}}");
            }
        }

        return r;
    }

private:

#define EXPR_ERROR(msg, expr, pos) msg << ", at pos " << (pos) << ": \"" << (expr) << "\""

    void BuildFrom(TStringBuf expr) {
        expr = StripString(expr);

        enum {
            NAME,
            VALUE,
        } tokenType = VALUE;
        TLabelBackend token;

        // only braces are left unescaped
        for (size_t i = 0; i != expr.size(); ++i) {
            char ch = expr[i];

            if (ch == '\\') {
                Y_ENSURE(i != expr.size() - 1, "unclosed escaping");

                token.append(UnescapeC(TStringBuf{expr.data() + i, 2}));
                ++i;
            } else if (ch == ' ' || (ch == '{' && tokenType == NAME) || (ch == '}' && tokenType == VALUE)) {
                ythrow yexception() << EXPR_ERROR("ill-formed substitution. unescaped '" << ch << "' inside a label "
                                    << (tokenType == NAME ? "name": "value"), expr, i);
            } else if ((ch == '{' && tokenType == VALUE) || (ch == '}' && tokenType == NAME)) {
                Y_ENSURE((i != expr.size() - 1) && (expr[i + 1] == ch),
                        EXPR_ERROR("ill-formed substitution. unescaped or unmatched braces '" << ch << "'", expr, i));

                ++i;

                switch (tokenType) {
                    case NAME: AppendName(token); break;
                    case VALUE: AppendValue(token); break;
                }

                token = {};
                tokenType = tokenType == NAME ? VALUE : NAME;
            } else {
                token += ch;
            }
        }

        if (!token.empty()) {
            switch (tokenType) {
                case NAME: ythrow yexception() << EXPR_ERROR("unclosed braces inside a substitution", expr, expr.size() - 1);
                case VALUE: AppendValue(token); break;
            }
        }
    }

#undef EXPR_ERROR

    bool IsLastElementAValue() const {
        return ValueNamePairs_.size() % 2 != 0;
    }

    void AppendValue(TStringBuf value) {
        if (!value.empty()) {
            TLabelsValidatorStrict::ValidateLabelValueSafe(value);
        }

        Y_ENSURE(!IsLastElementAValue(), "two label values cannot be inserted one after another");
        if (value.empty()) {
            ValueNamePairs_.emplace_back(std::nullopt);
        } else {
            ValueNamePairs_.emplace_back(value);
        }
    }

    void AppendName(TStringBuf name) {
        ValidateExternalLabelName(name);

        Y_ENSURE(IsLastElementAValue(), "two label names cannot be inserted one after another");
        ValueNamePairs_.emplace_back(name);
    }

private:
    TVector<std::optional<TLabelBackend>> ValueNamePairs_;
};

struct TLabelNameWithExpr {
    TLabelBackend Name;
    TLabelValueExpr Expr;
};

class TLabelsTransformer {
public:
    TLabelsTransformer(TLabels&& labels) {
        decltype(LabelsToDrop_) labelsToDrop;
        decltype(LabelsToAdd_) labelsToAdd;
        decltype(LabelsWithSubs_) labelsWithSubs;

        for (auto&& label: labels) {
            if (label.Value() == "-") {
                labelsToDrop.emplace(label.Name());
            } else if (!label.Value().Contains("{{")) {
                labelsToAdd.emplace(label.Name(), label.Value());
            } else {
                labelsWithSubs.emplace_back(TLabelNameWithExpr{
                    TLabelBackend{label.Name().data(), label.Name().size()},
                    label.Value()
                });
            }
        }

        LabelsToDrop_ = std::move(labelsToDrop);
        LabelsToAdd_ = std::move(labelsToAdd);
        LabelsWithSubs_ = std::move(labelsWithSubs);
    }

    void Transform(TLabelsHashMap& labels, TLabelsHashMap& optLabels) {
        // For rules that both change the value and rely on the old one.
        // For instance: "cluster=sdev -> project=newFixedValue, client={{project}}"
        TLabelsHashMap labelNameToOriginalValue;

        for (auto& label: LabelsWithSubs_) {
            if (!StoreLabelsOriginalValues(label.Expr.GetNames(), labels, optLabels, labelNameToOriginalValue)) {
                SA_LOG(WARN) << "could not retrieve all label values specified in the transformation"
                             << "\"" << label.Expr.ToString() << "\". Skipping";
                continue;
            }

            TLabelBackend newValue = label.Expr.Eval(labelNameToOriginalValue, labels, optLabels);
            labels[label.Name] = newValue;
        }

        for (auto& labelName: LabelsToDrop_) {
            labels.erase(labelName);
        }

        for (auto&& it: LabelsToAdd_) {
            labels[it.first] = it.second;
        }
    }

private:
    bool StoreLabelsOriginalValues(const TVector<TStringBuf>& names,
                                   const TLabelsHashMap& labels, const TLabelsHashMap& optLabels,
                                   TLabelsHashMap& labelNameToValue) const {
        for (auto& name: names) {
            if (!(labels.contains(name) || optLabels.contains(name))) {
                SA_LOG(WARN) << "label \"" << name << "\" is not present in original labels.";
                return false;
            }

            labelNameToValue[name] = labels.contains(name) ? labels.at(name) : optLabels.at(name);
        }

        return true;
    }

    TLabelsTransformer(THashSet<TLabelBackend>&& labelsToDrop, TLabelsHashMap&& labelsToAdd, TVector<TLabelNameWithExpr>&& labelsWithSubs)
        : LabelsToDrop_{labelsToDrop}
        , LabelsToAdd_{labelsToAdd}
        , LabelsWithSubs_{labelsWithSubs}
    {}

private:
    THashSet<TLabelBackend> LabelsToDrop_;
    TLabelsHashMap LabelsToAdd_;
    TVector<TLabelNameWithExpr> LabelsWithSubs_;
};

class TDataTransformer {
public:
    template <typename TValueType>
    std::pair<TInstant, TValueType> Transform(TInstant time, TValueType value) {
        // If you add support for ReplaceTs (ts transformation), beware that it can conflict with ts grid alignment
        // in pull modules. For more info, see a SOLOMON-3930 task
        return {time, value};
    }
};

template <typename TFunc>
void AddNewLabel(TFunc nameValidator, TFunc valueValidator, TStringBuf name, TStringBuf value, TLabels& result) {
    nameValidator(name);
    valueValidator(value);

    Y_ENSURE(result.Add(name, value),
            "multiple actions or matchers were defined with a label \"" << name << "\" inside one Rule");
}

template <typename TFunc>
TLabels ParseLabels(TStringBuf str, TFunc nameValidator, TFunc valueValidator) {
    TLabels result;

    TLabelBackend name;
    TLabelBackend value;
    // will switch from name to value back and forth
    TLabelBackend* token = &name;

    str = StripString(str);

    for (size_t i = 0; i < str.size(); ++i) {
        char ch = str[i];

        switch (ch) {
            case '\\': {
                Y_ENSURE(i != str.size() - 1, "unclosed escaping");

                auto nextCh = str[i + 1];
                if (nextCh == '{' || nextCh == '}') {
                    // Propagate braces to later LabelValueExpr validation and parsing
                    *token += '\\';
                    *token += nextCh;
                } else {
                    token->append(UnescapeC(TStringBuf{str.data() + i, 2}));
                }

                ++i;
                break;
            }
            case ' ':
                ythrow yexception() << "unescaped space characters";
            case ',':
                AddNewLabel(nameValidator, valueValidator, name, value, result);
                name = {};
                value = {};
                token = &name;

                while (i < str.size() - 1 && str[i + 1] == ' ') {
                    ++i;
                }

                break;
            case '=':
                Y_ENSURE(token != &value,
                        "multiple '=' inside a label value. Either remove it or escape it with '\\'");

                Y_ENSURE(!name.empty(), "empty label name");

                token = &value;
                break;
            default:
                *token += ch;
                break;
        }
    }

    if (!(name.empty() && value.empty())) {
        AddNewLabel(nameValidator, valueValidator, name, value, result);
    }

    return result;

}

class TTransformingConsumer: public ITransformingConsumer {
public:
    TTransformingConsumer(const TTransformationsConfig* transformations) {
        Rules_.reserve(transformations->RuleSize());

        for (auto& protoRule: transformations->GetRule()) {
            TRule rule;

            {
                TLabels matchLabels = protoRule.GetMatch().empty()
                    ? TLabels{} :
                    ParseMatchLabels(protoRule.GetMatch());

                rule.Match = CreateGlobMatcher(std::move(matchLabels));
            }

            if (!protoRule.GetReplaceMeta().empty()) {
                TLabels replaceMetaLabels = ParseReplaceMetaLabels(protoRule.GetReplaceMeta());

                rule.LabelsTransformer = TLabelsTransformer(std::move(replaceMetaLabels));

                NeedToTransformLabels_ = true;
            }

            if (!protoRule.GetReplaceData().empty()) {
                // TODO: parse
                NeedToTransformData_ = true;
            }

            if (!protoRule.GetReplaceTs().empty()) {
                // TODO: parse
                NeedToTransformData_ = true;
            }

            Y_ENSURE(NeedToTransformData_ || NeedToTransformLabels_, "no valid transformations were specified");

            Rules_.emplace_back(std::move(rule));
        }
    }

    void SetInnerConsumer(IMetricConsumer* consumer) override {
        Consumer_ = consumer;
    }

    void OnStreamBegin() override {
        State_ = EState::Root;
        Consumer_->OnStreamBegin();
    }
    void OnStreamEnd() override {
        Consumer_->OnStreamEnd();
        Consumer_ = nullptr;
    }

    void OnCommonTime(TInstant time) override {
        Consumer_->OnCommonTime(time);
    }

    void OnMetricBegin(NMonitoring::EMetricType type) override {
        State_ = EState::Metric;
        Consumer_->OnMetricBegin(type);
    }

    void OnMetricEnd() override {
        Consumer_->OnMetricEnd();
        State_ = EState::Root;
        HaveSeenAllLabels_ = false;
    }

    void OnLabelsBegin() override {
        if (State_ == EState::Root) {
            State_ = EState::CommonLabels;
        }
    }

    void OnLabelsEnd() override {
        if (State_ == EState::CommonLabels) {
            return;
        }

        if (Labels_.empty()) {
            return;
        }

        if (NeedToTransformLabels_) {
            TransformLabels();

            if (Labels_.empty()) {
                ythrow yexception() << "no labels were left after performing transformations";
            }
        }

        HaveSeenAllLabels_ = true;
        // if we've already buffered some data, transform it
        if (!ValuesBuffer_.Empty()) {
            TransformData();
            ValuesBuffer_.Clear();
        }

        Consumer_->OnLabelsBegin();
        for (auto it: Labels_) {
            Consumer_->OnLabel(it.first, it.second);
        }
        Consumer_->OnLabelsEnd();

        Labels_.clear();
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        auto& labels = (State_ == EState::Metric) ? Labels_ : CommonLabels_;
        labels.emplace(name, value);
    }

    template <typename TValueType>
    void OnValue(TInstant time, TValueType value) {
        if (!HaveSeenAllLabels_) {
            ValuesBuffer_.Add(time, value);
        } else {
            for (auto&& rule: Rules_) {
                if (rule.DataTransformer && rule.Match(Labels_, &CommonLabels_)) {
                    std::tie(time, value) = rule.DataTransformer->Transform(time, value);
                }
            }
        }

        if constexpr (std::is_same_v<TValueType, double>) {
            Consumer_->OnDouble(time, value);
        } else if constexpr (std::is_same_v<TValueType, i64>) {
            Consumer_->OnInt64(time, value);
        } else if constexpr (std::is_same_v<TValueType, ui64>) {
            Consumer_->OnUint64(time, value);
        } else if constexpr (std::is_same_v<TValueType, NMonitoring::IHistogramSnapshot*>) {
            Consumer_->OnHistogram(time, value);
        } else if constexpr (std::is_same_v<TValueType, NMonitoring::ISummaryDoubleSnapshot*>) {
            Consumer_->OnSummaryDouble(time, value);
        } else if constexpr (std::is_same_v<TValueType, NMonitoring::TLogHistogramSnapshot*>) {
            Consumer_->OnLogHistogram(time, value);
        } else {
            static_assert(TDependentFalse<TValueType>, "unsupported metric data type");
        }
    }

    void OnDouble(TInstant time, double value) override {
        OnValue(time, value);
    }

    void OnInt64(TInstant time, i64 value) override {
        OnValue(time, value);
    }

    void OnUint64(TInstant time, ui64 value) override {
        OnValue(time, value);
    }

    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

private:
    void TransformLabels() {
        for (auto&& rule: Rules_) {
            if (rule.LabelsTransformer && rule.Match(Labels_, &CommonLabels_)) {
                rule.LabelsTransformer->Transform(Labels_, CommonLabels_);
            }
        }
    }

    void TransformData() {
        // TODO: implement
    }

private:
    enum class EState {
        Root = 0,
        CommonLabels,
        Metric,
    };
    EState State_;

    IMetricConsumer* Consumer_;

    TLabelsHashMap CommonLabels_;
    TLabelsHashMap Labels_;

    bool HaveSeenAllLabels_ = false;

    struct TRule {
        THashLabelsMatcher Match;
        std::optional<TLabelsTransformer> LabelsTransformer;
        std::optional<TDataTransformer> DataTransformer;
        // TODO: support Action
    };

    bool NeedToTransformLabels_ = false;
    bool NeedToTransformData_ = false;
    TVector<TRule> Rules_;

    NMonitoring::TMetricTimeSeries ValuesBuffer_;
};

} // namespace

TLabels ParseMatchLabels(TStringBuf str) {
    return ParseLabels(str, ValidateMatchLabelName, ValidateMatchLabelValue);
}

TLabels ParseReplaceMetaLabels(TStringBuf str) {
    return ParseLabels(str, TLabelsValidator::ValidateLabelName, ValidateReplaceMetaLabelValue);
}

ITransformingConsumerPtr CreateTransformingConsumer(const TTransformationsConfig* transformations) {
    if (transformations && transformations->RuleSize() > 0) {
        return MakeHolder<TTransformingConsumer>(transformations);
    } else {
        return nullptr;
    }
}

} // namespace NSolomon::NAgent {
