#include <solomon/agent/lib/consumers/grouping_encoder.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NAgent;

TEST(TShardKeySubstitutionsTest, SubstituteAll) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "{{app_id}}"};

    {
        TRemovableLabels labels{
            {"cloud_id", "projectIsSubstituted"},
            {"folder_id", "clusterIsSubstituted"},
            {"app_id", "serviceIsSubstituted"},
            {"metric", "additionalLabel"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 1u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");

        ASSERT_EQ(shardKey.Project, "projectIsSubstituted");
        ASSERT_EQ(shardKey.Cluster, "clusterIsSubstituted");
        ASSERT_EQ(shardKey.Service, "serviceIsSubstituted");
    }

    {
        TRemovableLabels labels{
            // missing project, cluster, service --> metric will be skipped
            {"metric", "additionalLabel"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);
        ASSERT_TRUE(shardKeyOrError.Fail());

        ASSERT_EQ(labels.size(), 1u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");
    }

    {
        TRemovableLabels labels{};
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);
        ASSERT_TRUE(shardKeyOrError.Fail());

        ASSERT_EQ(labels.size(), 0u);
    }

    {
        TRemovableLabels labels{
            {"metric", "someMetricName"},
            {"cloud_id", "onlyProjectSubstitutionIsPresent"},
            {"cpu", "2"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);
        ASSERT_TRUE(shardKeyOrError.Fail());

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "someMetricName");
        ASSERT_EQ(labels[1].Name(), "cpu");
        ASSERT_EQ(labels[1].Value(), "2");
    }

    {
        TRemovableLabels labels{
            {"metric", "someMetricName"},
            {"folder_id", "onlyClusterSubstitutionIsPresent"},
            {"cpu", "2"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);
        ASSERT_TRUE(shardKeyOrError.Fail());

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "someMetricName");
        ASSERT_EQ(labels[1].Name(), "cpu");
        ASSERT_EQ(labels[1].Value(), "2");
    }

    {
        TRemovableLabels labels{
            {"metric", "someMetricName"},
            {"cpu", "2"},
            {"app_id", "onlyServiceSubstitutionIsPresent"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);
        ASSERT_TRUE(shardKeyOrError.Fail());

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "someMetricName");
        ASSERT_EQ(labels[1].Name(), "cpu");
        ASSERT_EQ(labels[1].Value(), "2");
    }
}

TEST(TShardKeySubstitutionsTest, RewriteAll) {
    TShardKeySubstitutions shardKeySubstitutions{
        "projectIsRewritten", "clusterIsRewritten", "serviceIsRewritten",
    };

    {
        TRemovableLabels labels{
            {"metric", "someMetricName"},
            {"cpu", "2"},
            {"app_id", "someAppId"},
        };

        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 3u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "someMetricName");
        ASSERT_EQ(labels[1].Name(), "cpu");
        ASSERT_EQ(labels[1].Value(), "2");
        ASSERT_EQ(labels[2].Name(), "app_id");
        ASSERT_EQ(labels[2].Value(), "someAppId");

        ASSERT_EQ(shardKey.Project, "projectIsRewritten");
        ASSERT_EQ(shardKey.Cluster, "clusterIsRewritten");
        ASSERT_EQ(shardKey.Service, "serviceIsRewritten");
    }
}

TEST(TShardKeySubstitutionsTest, RewriteAndSubstitute) {
    TShardKeySubstitutions shardKeySubstitutions{
        "projectIsRewritten", "{{some_cluster_label}}", "serviceIsRewritten",
    };

    {
        TRemovableLabels labels{
            {"some_cluster_label", "clusterIsSubstituted"},
            {"cpu", "2"},
            {"app_id", "someAppId"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "app_id");
        ASSERT_EQ(labels[0].Value(), "someAppId");
        ASSERT_EQ(labels[1].Name(), "cpu");
        ASSERT_EQ(labels[1].Value(), "2");

        ASSERT_EQ(shardKey.Project, "projectIsRewritten");
        ASSERT_EQ(shardKey.Cluster, "clusterIsSubstituted");
        ASSERT_EQ(shardKey.Service, "serviceIsRewritten");
    }
}

TEST(TShardKeySubstitutionsTest, CopyingAndBoolCast) {
    TShardKeySubstitutions shardKeySubstitutionsEmpty;
    ASSERT_FALSE(shardKeySubstitutionsEmpty);
    TShardKeySubstitutions shardKeySubstitutionsCopyFromEmpty = shardKeySubstitutionsEmpty;
    ASSERT_FALSE(shardKeySubstitutionsCopyFromEmpty);

    TShardKeySubstitutions shardKeySubstitutions{
        "projectIsRewritten", "{{some_cluster_label}}", "serviceIsRewritten",
    };
    ASSERT_TRUE(shardKeySubstitutions);
    TShardKeySubstitutions shardKeySubstitutionsCopy = shardKeySubstitutions;
    ASSERT_TRUE(shardKeySubstitutionsCopy);
}

TEST(TShardKeySubstitutionsTest, HostLabel) {
    {
        TShardKeySubstitutions shardKeySubstitutions{"prj", "clstr", "srvc", true};

        TRemovableLabels labels{
            {"metric", "additionalLabel"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");

        ASSERT_EQ(labels[1].Name(), "host");
        ASSERT_EQ(labels[1].Value(), "");

        ASSERT_EQ(shardKey.Project, "prj");
        ASSERT_EQ(shardKey.Cluster, "clstr");
        ASSERT_EQ(shardKey.Service, "srvc");
    }

    {
        TShardKeySubstitutions shardKeySubstitutions{"prj", "clstr", "srvc", false};

        TRemovableLabels labels{
            {"metric", "additionalLabel"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 1u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");

        ASSERT_EQ(shardKey.Project, "prj");
        ASSERT_EQ(shardKey.Cluster, "clstr");
        ASSERT_EQ(shardKey.Service, "srvc");
    }

    {
        TShardKeySubstitutions shardKeySubstitutions{"prj", "clstr", "srvc", true};

        TRemovableLabels labels{
            {"metric", "additionalLabel"},
            {"host", "localhost"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");

        ASSERT_EQ(labels[1].Name(), "host");
        ASSERT_EQ(labels[1].Value(), "localhost");

        ASSERT_EQ(shardKey.Project, "prj");
        ASSERT_EQ(shardKey.Cluster, "clstr");
        ASSERT_EQ(shardKey.Service, "srvc");
    }

    {
        TShardKeySubstitutions shardKeySubstitutions{"prj", "clstr", "srvc", false};

        TRemovableLabels labels{
            {"metric", "additionalLabel"},
            {"host", "localhost"},
        };
        auto shardKeyOrError = shardKeySubstitutions.Substitute(labels);

        ASSERT_TRUE(shardKeyOrError.Success());
        auto& shardKey = shardKeyOrError.Value();

        ASSERT_EQ(labels.size(), 2u);
        ASSERT_EQ(labels[0].Name(), "metric");
        ASSERT_EQ(labels[0].Value(), "additionalLabel");

        ASSERT_EQ(labels[1].Name(), "host");
        ASSERT_EQ(labels[1].Value(), "localhost");

        ASSERT_EQ(shardKey.Project, "prj");
        ASSERT_EQ(shardKey.Cluster, "clstr");
        ASSERT_EQ(shardKey.Service, "srvc");
    }
}

TEST(TGroupingEncoderTest, OneShardOneMetric) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "{{app_id}}"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};
    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();
    encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);

    encoder.OnLabelsBegin();

    TVector<NMonitoring::TLabel> labels{
        {"cloud_id", "ProjectValue"}, {"folder_id", "ClusterValue"}, {"app_id", "ServiceValue"},
        {"metric", "someMetricName"}
    };
    for (auto&& l: labels) {
        encoder.OnLabel(l.Name(), l.Value());
    }

    encoder.OnLabelsEnd();

    encoder.OnUint64(TInstant::Zero(), 123);

    encoder.OnMetricEnd();
    encoder.OnStreamEnd();
    encoder.Close();

    auto shardsData = groupingEncoder.GetShardsData();

    ASSERT_EQ(shardsData.size(), 1u);
    ASSERT_EQ(shardsData[0].first.Project, "ProjectValue");
    ASSERT_EQ(shardsData[0].first.Cluster, "ClusterValue");
    ASSERT_EQ(shardsData[0].first.Service, "ServiceValue");

    ASSERT_EQ(shardsData[0].second, R"({"sensors":[{"kind":"GAUGE","labels":{"metric":"someMetricName"},"value":123}]})");
}

TEST(TGroupingEncoderTest, DifferentShardsWithOneMetric) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "clusterIsRewritten", "{{app_id}}"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};

    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();

        TVector<NMonitoring::TLabel> labels{
            {"cloud_id", "project1"}, {"app_id", "service1"},
            {"metric", "metric1"}
        };
        for (auto&& l: labels) {
            encoder.OnLabel(l.Name(), l.Value());
        }

        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::COUNTER);
        encoder.OnLabelsBegin();

        TVector<NMonitoring::TLabel> labels{
            {"cloud_id", "project2"}, {"app_id", "service2"},
            {"metric", "metric2"}
        };
        for (auto&& l: labels) {
            encoder.OnLabel(l.Name(), l.Value());
        }

        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 456);
        encoder.OnMetricEnd();
    }

    encoder.OnStreamEnd();
    encoder.Close();

    auto shardsData = groupingEncoder.GetShardsData();

    ASSERT_EQ(shardsData.size(), 2u);

    ASSERT_EQ(shardsData[0].first.Project, "project1");
    ASSERT_EQ(shardsData[0].first.Cluster, "clusterIsRewritten");
    ASSERT_EQ(shardsData[0].first.Service, "service1");
    ASSERT_EQ(shardsData[0].second, R"({"sensors":[{"kind":"GAUGE","labels":{"metric":"metric1"},"value":123}]})");

    ASSERT_EQ(shardsData[1].first.Project, "project2");
    ASSERT_EQ(shardsData[1].first.Cluster, "clusterIsRewritten");
    ASSERT_EQ(shardsData[1].first.Service, "service2");
    ASSERT_EQ(shardsData[1].second, R"({"sensors":[{"kind":"COUNTER","labels":{"metric":"metric2"},"value":456}]})");
}

TEST(TGroupingEncoderTest, DifferentShardsWithOneMetricSOLOMON3993) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "compute"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};

    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();

        TVector<NMonitoring::TLabel> labels{
            {"cloud_id", "aoe9shbqc2v314v7fp3d"},
            {"resource_id", "c8r41bs4dkictkok8dso"},
            {"folder_id", "aoed5i52uquf5jio0oec"},
        };
        for (auto&& l: labels) {
            encoder.OnLabel(l.Name(), l.Value());
        }

        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();

        TVector<NMonitoring::TLabel> labels{
            {"cloud_id", "aoe96r0tq4cq6gf6i2no"},
            {"resource_id", "c8r4dg61cn7s0tjaj5mh"},
            {"folder_id", "aoeqn6r8hruqkoq6cct8"},
        };
        for (auto&& l: labels) {
            encoder.OnLabel(l.Name(), l.Value());
        }

        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 456);
        encoder.OnMetricEnd();
    }

    encoder.OnStreamEnd();
    encoder.Close();

    auto shardsData = groupingEncoder.GetShardsData();

    ASSERT_EQ(shardsData.size(), 2u);

    ASSERT_EQ(shardsData[0].first.Project, "aoe96r0tq4cq6gf6i2no");
    ASSERT_EQ(shardsData[0].first.Cluster, "aoeqn6r8hruqkoq6cct8");
    ASSERT_EQ(shardsData[0].first.Service, "compute");
    ASSERT_EQ(shardsData[0].second, R"({"sensors":[{"kind":"GAUGE","labels":{"resource_id":"c8r4dg61cn7s0tjaj5mh"},"value":456}]})");

    ASSERT_EQ(shardsData[1].first.Project, "aoe9shbqc2v314v7fp3d");
    ASSERT_EQ(shardsData[1].first.Cluster, "aoed5i52uquf5jio0oec");
    ASSERT_EQ(shardsData[1].first.Service, "compute");
    ASSERT_EQ(shardsData[1].second, R"({"sensors":[{"kind":"GAUGE","labels":{"resource_id":"c8r41bs4dkictkok8dso"},"value":123}]})");
}

TEST(TGroupingEncoderTest, MetricValuesBeforeLabels) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "{{app_id}}"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};

    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();
    encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);

    ASSERT_THROW(encoder.OnUint64(TInstant::Zero(), 123), yexception);
}

TEST(TGroupingEncoderTest, MetricValuesBeforeLabelsOnSecondMetric) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "{{app_id}}"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};

    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();
        encoder.OnLabel("someLabel", "someValue");
        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        ASSERT_THROW(encoder.OnUint64(TInstant::Zero(), 123), yexception);
    }
}

TEST(TGroupingEncoderTest, MetricsWithMissingSubstitutions) {
    TShardKeySubstitutions shardKeySubstitutions{"{{cloud_id}}", "{{folder_id}}", "{{app_id}}"};
    TGroupingEncoder groupingEncoder{shardKeySubstitutions, JsonEncoderFactory};

    NMonitoring::IMetricEncoder& encoder = groupingEncoder;

    encoder.OnStreamBegin();

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();
        encoder.OnLabel("cloud_id", "cloud");
        encoder.OnLabel("folder_id", "folder");
        encoder.OnLabel("app_id", "app");
        encoder.OnLabel("someLabel", "someValue");
        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();
        encoder.OnLabel("cloud_id", "cloud");
        encoder.OnLabel("folder_id", "cloud");
        encoder.OnLabel("missing", "substitutions");
        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();
        encoder.OnLabel("cloud_id", "cloud");
        encoder.OnLabel("missing", "substitutions");
        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    {
        encoder.OnMetricBegin(NMonitoring::EMetricType::GAUGE);
        encoder.OnLabelsBegin();
        encoder.OnLabel("missing", "substitutions");
        encoder.OnLabelsEnd();
        encoder.OnUint64(TInstant::Zero(), 123);
        encoder.OnMetricEnd();
    }

    encoder.OnStreamEnd();
    encoder.Close();

    auto shardsData = groupingEncoder.GetShardsData();
    ASSERT_EQ(shardsData.size(), 1u); // because one metric was skipped

    ASSERT_EQ(shardsData[0].first.Project, "cloud");
    ASSERT_EQ(shardsData[0].first.Cluster, "folder");
    ASSERT_EQ(shardsData[0].first.Service, "app");
    ASSERT_EQ(shardsData[0].second, R"({"sensors":[{"kind":"GAUGE","labels":{"someLabel":"someValue"},"value":123}]})");
}

TEST(TShardKeyTest, TShardKey) {
    TShardKey shardKey{"projectValue", "clusterValue", "serviceValue"};

    ASSERT_EQ(shardKey.Project, "projectValue");
    ASSERT_EQ(shardKey.Cluster, "clusterValue");
    ASSERT_EQ(shardKey.Service, "serviceValue");

    TShardKey shardKey2{"projectValue", "clusterValue", "serviceValue"};
    ASSERT_TRUE(shardKey == shardKey2);
}
