#include <solomon/agent/lib/consumers/transforming_consumer.h>
#include <solomon/agent/lib/storage/stub/stub_storage.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <google/protobuf/text_format.h>

using namespace NSolomon::NAgent;

namespace {

TTransformationsConfig CreateConfig(const TString& str) {
    TTransformationsConfig config;
    google::protobuf::TextFormat::ParseFromString(str, &config);
    return config;
}

void FillConsumer(
        NMonitoring::IMetricConsumer* consumer,
        const TLabels& commonLabels,
        const TVector<TLabels>& metricsLabels)
{
    consumer->OnStreamBegin();

    consumer->OnLabelsBegin();
    for (auto&& label: commonLabels) {
        consumer->OnLabel(TString{label.Name()}, TString{label.Value()});
    }
    consumer->OnLabelsEnd();

    for (auto&& metricLabels: metricsLabels) {
        consumer->OnMetricBegin(NMonitoring::EMetricType::COUNTER);

        consumer->OnLabelsBegin();
        for (auto&& label: metricLabels) {
            consumer->OnLabel(TString{label.Name()}, TString{label.Value()});
        }
        consumer->OnLabelsEnd();

        consumer->OnUint64(TInstant::Now(), 1234);

        consumer->OnMetricEnd();
    }

    consumer->OnStreamEnd();
}

} // namespace

TEST(TParseLabelsTest, MatchLabelsValid) {
    {
        auto labels = ParseMatchLabels("cluster=*,  service=*");

        ASSERT_EQ(labels.size(), 2u);

        ASSERT_EQ(labels[0].Name(), "cluster");
        ASSERT_EQ(labels[0].Value(), "*");

        ASSERT_EQ(labels[1].Name(), "service");
        ASSERT_EQ(labels[1].Value(), "*");
    }

    {
        auto labels = ParseMatchLabels(R"(someLabel=valueWithBraces{})");

        ASSERT_EQ(labels.size(), 1u);

        ASSERT_EQ(labels[0].Name(), "someLabel");
        ASSERT_EQ(labels[0].Value(), "valueWithBraces{}");
    }

    {
        TLabels labels;
        ASSERT_NO_THROW(labels = ParseMatchLabels("validLabel=validValue,")) << "one trailing comma is allowed";

        ASSERT_EQ(labels.size(), 1u);

        ASSERT_EQ(labels[0].Name(), "validLabel");
        ASSERT_EQ(labels[0].Value(), "validValue");
    }

    // Escaping
    {
        TLabels labels;
        ASSERT_NO_THROW(labels = ParseMatchLabels(R"(some\==label\,with\ characters, smth\ =\ smth)"))
                << "escaped characters";

        ASSERT_EQ(labels.size(), 2u);

        ASSERT_EQ(labels[0].Name(), "some=");
        ASSERT_EQ(labels[0].Value(), "label,with characters");

        ASSERT_EQ(labels[1].Name(), "smth ");
        ASSERT_EQ(labels[1].Value(), " smth");
    }

    {
        TLabels labels;
        ASSERT_NO_THROW(labels = ParseMatchLabels(R"(some\\Label=val\\\\ue)")) << "escaped characters";

        ASSERT_EQ(labels.size(), 1u);
        ASSERT_EQ(labels[0].Name(), "some\\Label");
        ASSERT_EQ(labels[0].Value(), "val\\\\ue");
    }
}

TEST(TParseLabelsTest, MatchLabelsInvalid) {
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("validLabel=validValue,,"), yexception, "Label name in a Match is empty or has non-printable characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels(",validLabel=validValue"), yexception, "Label name in a Match is empty or has non-printable characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("emptyValue="), yexception, "Label value in a Match is empty or has non-printable characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("multiple=equal, sign=inside=label, smth=smth"), yexception, "multiple '='");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("=emptyName"), yexception, "empty label name");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("="), yexception, "empty label name");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels(","), yexception, "Label name in a Match is empty or has non-printable characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels("label = value"), yexception, "unescaped space characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels(R"(some\tlabel=value)"), yexception, "Label name in a Match is empty or has non-printable characters");
    ASSERT_THROW_MESSAGE_HAS_SUBSTR(ParseMatchLabels(R"(some\nlabel=value)"), yexception, "Label name in a Match is empty or has non-printable characters");
}

TEST(TParseLabelsTest, ReplaceMetaLabelsValid) {
    {
        auto labels = ParseMatchLabels(" project=-, projectId=some-{old}-value ");

        ASSERT_EQ(labels.size(), 2u);

        ASSERT_EQ(labels[0].Name(), "project");
        ASSERT_EQ(labels[0].Value(), "-");

        ASSERT_EQ(labels[1].Name(), "projectId");
        ASSERT_EQ(labels[1].Value(), "some-{old}-value");
    }

    {
        auto labels = ParseMatchLabels(R"( project=-, projectId=some-{old\{value\}}-with-braces )");

        ASSERT_EQ(labels.size(), 2u);

        ASSERT_EQ(labels[0].Name(), "project");
        ASSERT_EQ(labels[0].Value(), "-");

        ASSERT_EQ(labels[1].Name(), "projectId");
        ASSERT_EQ(labels[1].Value(), R"(some-{old\{value\}}-with-braces)");
    }
}

TEST(TTransformingConsumerTest, CheckLabelsTransformations) {
    {
        // What is tested:
        // - White spaces in rules configurations are processed correctly
        // - That rules are chained ==> the next Rule can rely on a __result__ of a previous Rule
        // - Glob and exact matching
        // - Labels addition and removal
        // - Labels substitutions
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "clusterId=old-{{cluster}}-value,    serviceId=old-{{service}}-value, cluster=-,         service=-,     newLabel=fixedValue"
            }

            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "newLabel=fixedValue"
            }

            Rule {
                Match: "clusterId=*, serviceId=old-commonService-value"
                ReplaceMeta: "newLabel2={{newLabel}}2"
            }

            Rule {
                Match: "metric=two"
                ReplaceMeta: "found=metric_two"
            }

            Rule {
                ReplaceMeta: "newLabel3=worksWithEmptyMatch"
            }
        )");

        NTest::TStubStorage storage;
        IStorageMetricsConsumerPtr storageConsumer = storage.CreateShardConsumer("project", "service");
        ITransformingConsumerPtr consumer = CreateTransformingConsumer(&config);
        consumer->SetInnerConsumer(storageConsumer.Get());

        // Writing data
        FillConsumer(
            consumer.Get(),
            { {"cluster", "commonCluster"}, {"service", "commonService"} },
            {
                { {"metric", "one"} },
                { {"metric", "two"} },
            }
        );

        storageConsumer->Flush();

        // Reading data
        NTest::TChunks chunks = storage.Chunks();
        NTest::TChunk& chunk = chunks[0];

        bool isMetricOnePresent = false;
        bool isMetricTwoPresent = true;

        for (auto&& metric: chunk) {
            auto& labels = metric.Labels;

            {
                TMaybe<TLabel> label = labels.Find("metric");
                ASSERT_TRUE(label.Defined());
                ASSERT_TRUE(label->Value() == "one" || label->Value() == "two");

                if (label->Value() == "one") {
                    isMetricOnePresent = true;
                }

                if (label->Value() == "two") {
                    isMetricTwoPresent = true;
                }
            }

            {
                ASSERT_FALSE(labels.Find("cluster"));
                ASSERT_FALSE(labels.Find("service"));
            }

            {
                TMaybe<TLabel> label;

                label = labels.Find("clusterId");
                ASSERT_TRUE(label.Defined());
                ASSERT_EQ(label->Value(), "old-commonCluster-value");

                label = labels.Find("serviceId");
                ASSERT_TRUE(label.Defined());
                ASSERT_EQ(label->Value(), "old-commonService-value");
            }

            {
                // Despite of the existence of __two__ rules, new label is added only once
                ui64 newLabelCnt = 0;
                for (auto&& label: labels) {
                    if (label.Name() == "newLabel") {
                        ++newLabelCnt;
                        ASSERT_EQ(label.Value(), "fixedValue");
                    }
                }

                ASSERT_EQ(newLabelCnt, 1u);
            }

            // Are rules chained?
            {
                TMaybe<TLabel> label = labels.Find("newLabel2");
                ASSERT_TRUE(label.Defined());
                ASSERT_EQ(label->Value(), "fixedValue2");
            }

            {
                TMaybe<TLabel> label = labels.Find("metric");
                ASSERT_TRUE(label.Defined());

                if (label->Value() == "two") {
                    TMaybe<TLabel> foundLabel = labels.Find("found");
                    ASSERT_TRUE(foundLabel.Defined());
                    ASSERT_EQ(foundLabel->Value(), "metric_two");
                }
            }

            {
                TMaybe<TLabel> label = labels.Find("newLabel3");
                ASSERT_TRUE(label.Defined());
                ASSERT_EQ(label->Value(), "worksWithEmptyMatch");
            }
        }

        ASSERT_TRUE(isMetricOnePresent);
        ASSERT_TRUE(isMetricTwoPresent);
    }
}

TEST(TTransformingConsumerTest, InvalidRules) {
    {
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "cluster=-, cluster=123"
            }
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(CreateTransformingConsumer(&config), yexception,
                "multiple actions or matchers were defined with a label \"cluster\" inside one Rule");
    }

    {
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "cluster=123, cluster={{someValue}}"
            }
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(CreateTransformingConsumer(&config), yexception,
                "multiple actions or matchers were defined with a label \"cluster\" inside one Rule");
    }

    {
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, cluster=123"
                ReplaceMeta: "newName=newValue"
            }
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(CreateTransformingConsumer(&config), yexception,
                "multiple actions or matchers were defined with a label \"cluster\" inside one Rule");
    }

    {
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "cluster="
            }
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(CreateTransformingConsumer(&config), yexception,
                "Label value in a ReplaceMeta is empty or has non-printable characters");
    }

    {
        auto config = CreateConfig(R"(
            Rule {
                Match: "cluster=*, service=*"
                ReplaceMeta: "cluster=, project="
            }
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(CreateTransformingConsumer(&config), yexception,
                "Label value in a ReplaceMeta is empty or has non-printable characters");
    }
}
