#include "context.h"

namespace NSolomon::NAgent {
namespace {

class TAgentContext: public IAgentContext {
public:
    TAgentContext() = default;

    TAgentContext(
            const TAgentConfig& config,
            NMonitoring::TMetricRegistry& registry,
            TTimerThread& timerThread,
            IThreadPoolProviderPtr threadPoolProvider)
    {
        Cluster_ = config.GetCluster();
        BindBeforeConnect_ = config.GetBindBeforeConnect();

        if (config.HasAuthProvider()) {
            // TODO: move to the auth lib
            ConstructAuthMethods(config.GetAuthProvider(), registry, timerThread, std::move(threadPoolProvider));
        }

        if (config.HasRegistration()) {
            RememberAuthMethodsInRegistration(config.GetRegistration());
        }
    }

    NSolomon::IAuthProviderPtr GetAuthProvider(TString name) const override {
        if (auto it = AuthMethodToAuthProvider_.find(name); it != AuthMethodToAuthProvider_.end()) {
            return it->second;
        }

        return nullptr;
    }

    NSolomon::IAuthProviderPtr GetAuthProvider(EClusterType cluster) const override {
        if (auto it = ClusterToAuthProvider_.find(cluster); it != ClusterToAuthProvider_.end()) {
            return it->second;
        }

        return nullptr;
    }

    TString GetCluster() const override {
        return Cluster_;
    }

    bool DoBindBeforeConnect() const override {
        return BindBeforeConnect_;
    }

private:
    void ConstructAuthMethods(
            const TAuthProvider& authProviderConf,
            NMonitoring::TMetricRegistry& registry,
            TTimerThread& timerThread,
            IThreadPoolProviderPtr threadPoolProvider)
    {
        auto threadPool = authProviderConf.GetThreadPoolName()
                          ? threadPoolProvider->GetThreadPool(authProviderConf.GetThreadPoolName())
                          : threadPoolProvider->GetDefaultPool();

        for (auto& authMethodConfig: authProviderConf.GetAuthMethods()) {
            const TString& methodName = authMethodConfig.GetName();

            Y_ENSURE(!methodName.empty(), "auth method name cannot be empty");

            auto authProvider = CreateAuthProvider(authMethodConfig, registry, timerThread, threadPool);
            if (!authProvider) {
                ythrow yexception() << "no actual auth config is specified for auth method \"" << methodName << "\"";
            }

            bool isNew = AuthMethodToAuthProvider_.emplace(methodName, authProvider).second;
            Y_ENSURE(isNew, "auth method \"" << methodName << "\" is specified multiple times");
        }
    }

    void RememberAuthMethodsInRegistration(const TProviderRegistrationConfig& regConf) {
        for (const auto& endpoint: regConf.GetEndpoints()) {
            const auto cluster = endpoint.GetType();
            const auto& authMethod = endpoint.GetAuthMethod();

            if (cluster != EClusterType::UNKNOWN_CLUSTER && !authMethod.empty()) {
                auto it = AuthMethodToAuthProvider_.find(authMethod);
                Y_ENSURE(
                        it != AuthMethodToAuthProvider_.end(),
                        "non-existent auth method \"" << authMethod << "\" is specified inside the Registration config");

                ClusterToAuthProvider_[cluster] = it->second;
            }
        }
    }

private:
    TString Cluster_;
    bool BindBeforeConnect_{false};
    THashMap<TString, NSolomon::IAuthProviderPtr> AuthMethodToAuthProvider_;
    THashMap<EClusterType, NSolomon::IAuthProviderPtr> ClusterToAuthProvider_;
};

std::unique_ptr<TAgentContext> AGENT_CONTEXT = std::make_unique<TAgentContext>();

} // namespace

/**
 * Constructs a global object for storing data relevant for all parts of the application. Is not a singleton for
 * performance purposes. DO NOT call this procedure more than once during the app lifetime
 */
const IAgentContext* ConstructAgentCtx(
        const TAgentConfig& config,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timerThread,
        IThreadPoolProviderPtr threadPoolProvider)
{
    AGENT_CONTEXT = std::make_unique<TAgentContext>(config, registry, timerThread, std::move(threadPoolProvider));
    return AGENT_CONTEXT.get();
}

const IAgentContext* GetAgentCtx() {
    return AGENT_CONTEXT.get();
}

} // namespace NSolomon::NAgent
