#include "server.h"

#include <solomon/agent/lib/http/headers.h>
#include <solomon/agent/lib/thread/pool_provider.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/protos/http_server_config.pb.h>

#include <library/cpp/http/server/http_ex.h>
#include <library/cpp/json/json_writer.h>

#include <util/datetime/cputimer.h>
#include <util/stream/length.h>

using TLibraryHttpServer = ::THttpServer;
using TMyHttpServer = ::NSolomon::NAgent::THttpServer;

namespace NSolomon {
namespace NAgent {
namespace {

TStringBuf ErrorMessageWithoutLocation(TStringBuf errorMsg) {
    // "bin/agent.cpp:133: some error msg: reason" --> "some error msg: reason"
    return errorMsg.After(':').After(':');
}

} // namespace

///////////////////////////////////////////////////////////////////////////////
// THttpServer::TImpl
///////////////////////////////////////////////////////////////////////////////
class TMyHttpServer::TImpl:
        public TLibraryHttpServer,
        public TLibraryHttpServer::ICallBack
{
    class THttpReplier: public THttpClientRequestEx {
    public:
        THttpReplier(const TMyHttpServer::TImpl* server)
            : ServerImpl_(server)
        {
        }

    private:
        TString GetRemoteAddr() const {
            NAddr::TOpaqueAddr peerAddr;
            int rc = getpeername(Socket(), peerAddr.MutableAddr(), peerAddr.LenPtr());
            return rc == 0 ? PrintHost(peerAddr) : "unknown address";
        }

        bool Reply(void*) final {
            TSimpleTimer timer;

            if (!ProcessHeaders()) {
                return true;
            }
            RD.Scan();
            THttpResponse resp;

            IHttpHandler* handler = ServerImpl_->FindHandler(RD.ScriptName());

            if (handler != nullptr) {
                try {
                    THttpRequest req{Input(), RD, Buf};
                    TStringBuf method = TStringBuf(Input().FirstLine()).Before(' ');

                    ServerImpl_->OnRequest(RD.ScriptName(), req.Body.Size());

                    if (method == TStringBuf("GET")) {
                        handler->OnGet(req, &resp);
                    } else if (method == TStringBuf("POST")) {
                        handler->OnPost(req, &resp);
                    } else {
                        resp.SetHttpCode(HTTP_METHOD_NOT_ALLOWED);
                    }
                } catch (const THttpError& e) {
                    resp.SetHttpCode(e.Code());
                    if (e.Code() >= HTTP_BAD_REQUEST) {
                        SA_LOG(ERROR) << e.what();
                        resp.SetContentType(TStringBuf("application/json"));
                        resp.SetContent(ToJsonError(ErrorMessageWithoutLocation(e.AsStrBuf())));
                    }
                } catch (const yexception& e) {
                    SA_LOG(ERROR) << e.what();
                    resp.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
                    resp.SetContentType(TStringBuf("application/json"));
                    resp.SetContent(ToJsonError(ErrorMessageWithoutLocation(e.AsStrBuf())));
                } catch (...) {
                    const TString& msg = CurrentExceptionMessage();

                    SA_LOG(ERROR) << msg;
                    resp.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
                    resp.SetContentType(TStringBuf("application/json"));
                    resp.SetContent(ToJsonError(ErrorMessageWithoutLocation(TStringBuf{msg})));
                }
            } else {
                TString message = "Unknown path ";
                message += RD.ScriptName();
                resp.SetHttpCode(HTTP_NOT_FOUND);
                resp.SetContent(ToJsonError(message));
                resp.SetContentType(TStringBuf("application/json"));
            }

            resp.AddHeader("Server", SERVER_HEADER);
            resp.AddHeader("X-Content-Type-Options", "nosniff");

            TDuration time;
            if (handler != nullptr) {
                TCountingOutput out{&Output()};
                resp.OutTo(out);
                time = timer.Get();

                ServerImpl_->OnRequestCompleted(RD.ScriptName(), resp.HttpCode(), time, out.Counter());
            } else {
                resp.OutTo(Output());
                time = timer.Get();
            }

            SA_LOG(INFO)
                << GetRemoteAddr() << ' ' << Input().FirstLine() << ' '
                << static_cast<int>(resp.HttpCode())
                << " took: " << time.MilliSeconds() << "ms";

            return true;
        }

        TString ToJsonError(TStringBuf message) {
            TString json;
            {
                TStringOutput output(json);
                NJson::TJsonWriter writer(&output, false);
                writer.OpenMap();
                writer.Write(TStringBuf("error"), message);
                writer.CloseMap();
                writer.Flush();
            }
            return json;
        }

    private:
        const TMyHttpServer::TImpl* ServerImpl_;
    };

public:
    explicit TImpl(const THttpServerOptions& options, TSimpleSharedPtr<IThreadPool> pool, IServerStatusListener* statusListener)
        : TLibraryHttpServer(this, /* mainWorkers = */pool, /* failWorkers = */pool, options)
        , StatusListener_(statusListener)
    {
    }

    TClientRequest* CreateClient() override {
        return new THttpReplier(this);
    }

    IHttpHandler* FindHandler(TStringBuf path) const {
        auto* handler = Handlers_.FindPtr(path);

        if (handler) {
            return handler->Get();
        }

        return nullptr;
    }

    void OnRequest(TStringBuf path, ui64 payloadSize) const {
        if (StatusListener_ != nullptr) {
            StatusListener_->OnRequest(path, payloadSize);
        }
    }

    void OnRequestCompleted(TStringBuf path, HttpCodes code, TDuration time, ui64 responseBytes) const {
        if (StatusListener_ != nullptr) {
            StatusListener_->OnRequestCompleted(path, code, time, responseBytes);
        }
    }

    void AddHandler(TStringBuf path, IHttpHandlerPtr handler) {
        TString pathStr{path};
        Y_ENSURE(FindHandler(pathStr) == nullptr, "duplicate path: " << pathStr);
        Handlers_.emplace(pathStr, std::move(handler));
    }

private:
    THashMap<TString, IHttpHandlerPtr> Handlers_;
    IServerStatusListener* StatusListener_;
};

///////////////////////////////////////////////////////////////////////////////
// THttpServer
///////////////////////////////////////////////////////////////////////////////

THttpServer::THttpServer(const THttpServerConfig& config, IThreadPoolProvider& threadPoolProvider, IServerStatusListener* statusListener) {
    THttpServerOptions options;
    options.AddBindAddress(config.GetBindAddress(), config.GetBindPort());
    options.SetMaxConnections(config.GetMaxConnections());
    options.SetOutputBufferSize(config.GetOutputBufferSize());
    options.EnableKeepAlive(true);
    options.EnableCompression(false);

    const TString& poolName = config.GetThreadPoolName();
    TSimpleSharedPtr<IThreadPool> pool = poolName
        ? threadPoolProvider.GetThreadPool(poolName)
        : threadPoolProvider.GetDefaultPool();

    Impl_.Reset(new TImpl(options, pool, statusListener));
}

THttpServer::~THttpServer() {
    Stop();
}

void THttpServer::Start() {
    Y_ENSURE(Impl_->Start(), "Could not start server, error " << Impl_->GetErrorCode() << ": " << Impl_->GetError());
    THttpServerOptions::TAddr addr = Impl_->Options().BindSockaddr.front();
    SA_LOG(INFO) << "Start http server on http://"
                 << addr.Addr << ':' << addr.Port << '/';
}

void THttpServer::Stop() {
    Impl_->Stop();
}

void THttpServer::AddHandler(TStringBuf path, IHttpHandlerPtr handler) {
    Impl_->AddHandler(path, std::move(handler));
}

} // namespace NAgent
} // namespace NSolomon
