#include "data_puller.h"

#include <solomon/agent/lib/consumers/transforming_consumer.h>
#include <solomon/agent/lib/selfmon/status/puller_status.h>
#include <solomon/agent/misc/background_threads.h>
#include <solomon/agent/misc/countdown_event.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/timer_thread.h>

#include <library/cpp/monlib/encode/protobuf/protobuf.h>

#include <util/digest/multi.h>
#include <util/generic/scope.h>

namespace NSolomon {
namespace NAgent {

using IThreadPoolPtr = TSimpleSharedPtr<IThreadPool>;

namespace {

///////////////////////////////////////////////////////////////////////////////
// TDefaultDataPuller
///////////////////////////////////////////////////////////////////////////////
class TDefaultDataPuller: public IDataPuller {
private:
    using TDefaultDataPullerPtr = TIntrusivePtr<TDefaultDataPuller>;

public:
    TDefaultDataPuller(IThreadPool* pool, TPullerStatus* status, TDuration gcDelay = TDuration::Zero())
        : Timer_("PullScheduler")
        , ThreadPool_(pool) // Note: pool has to be already started!
        , Status_(status)
        , GcDelay_(gcDelay)
    {
    }

    ~TDefaultDataPuller() {
        try {
            Stop();
        } catch (...) {}
    }

private:
    void Start() override {
        RunningTasksCountdown_ = {};
        Timer_.Start();
    }

    void Stop() override {
        CancelTasks();

        RunningTasksCountdown_.Stop();
        Timer_.Stop();

        while (!RunningTasksCountdown_.Await(TDuration::Seconds(10))) {
            SA_LOG(DEBUG) << "waiting for pull tasks completion";
        }
    }

    class TPullTask: public TMoveOnly {
    public:
        TPullTask(
                TDefaultDataPullerPtr dataPuller,
                IStorageConsumerProviderPtr storageConsumerProvider,
                IPullModulePtr module,
                TDuration period,
                const TTransformationsConfig* transformations)
            : DataPuller_{dataPuller}
            , StorageConsumerProvider_{std::move(storageConsumerProvider)}
            , Module_{module}
            , Period_{period}
        {
            TransformingConsumer_ = CreateTransformingConsumer(transformations);
        }

        TPullTask(TPullTask&& other)
            : State_{other.State_.load()}
            , DataPuller_{std::exchange(other.DataPuller_, nullptr)}
            , StorageConsumerProvider_{std::exchange(other.StorageConsumerProvider_, nullptr)}
            , Module_{std::exchange(other.Module_, nullptr)}
            , Period_{other.Period_}
            , TransformingConsumer_{std::exchange(other.TransformingConsumer_, nullptr)}
        {
        }

        void operator()(const TTaskContext& ctx) {
            EPullTaskState expected = EPullTaskState::Waiting;
            if (!State_.compare_exchange_strong(expected, EPullTaskState::Running)) {
                SA_LOG(WARN) << "skipping pull from " << Module_->Name()
                             << " because the previous call is still in progress";

                DataPuller_->OnTaskSkipped(*Module_);
                return;
            }

            if (!DataPuller_->RunningTasksCountdown_.TryInc()) {
                // Counter is already stopped
                State_ = EPullTaskState::Waiting;
                return;
            }

            Y_SCOPE_EXIT(this) {
                State_ = EPullTaskState::Waiting;
                DataPuller_->RunningTasksCountdown_.Dec();
            };

            TMaybe<TString> errorMessage;
            int returnValue = 0;
            TDuration execTime;

            const auto waitTime = TInstant::Now() - ctx.Scheduled;
            if (waitTime >= Period_) {
                SA_LOG(WARN) << "skipping pull from " << Module_->Name() << " because period is " << Period_
                             << " but wait time is " << waitTime;

                DataPuller_->OnTaskSkipped(*Module_, waitTime);

                return;
            }

            try {
                auto now = TInstant::Now();
                SA_LOG(DEBUG) << "start pulling from " << Module_->Name();

                NMonitoring::IMetricConsumer* consumer;

                TInstant defaultTs = TInstant::Seconds(now.Seconds() - (now.Seconds() % Period_.Seconds()));
                IStorageMetricsConsumerPtr storageConsumer = StorageConsumerProvider_->CreateConsumer(defaultTs);

                if (TransformingConsumer_) {
                    // If you add support for ReplaceTs (ts transformation), beware that it can conflict with ts grid
                    // alignment in pull modules. For more info, see a SOLOMON-3930 task
                    TransformingConsumer_->SetInnerConsumer(storageConsumer.Get());
                    consumer = TransformingConsumer_.Get();
                } else {
                    consumer = storageConsumer.Get();
                }

                returnValue = Module_->Pull(defaultTs, consumer);

                execTime = TInstant::Now() - now;
                SA_LOG(DEBUG) << "pulling from " << Module_->Name() << " took "
                              << execTime << ", return code: " << returnValue;

                storageConsumer->Flush();

                DataPuller_->OnTaskCompleted(*Module_, returnValue, execTime, errorMessage);

                if (returnValue != 0) {
                    DataPuller_->Cancel(*Module_);
                }
            } catch (...) {
                const TString& msg = CurrentExceptionMessage();
                SA_LOG(ERROR) << "error while pulling " << Module_->Name() << ": "
                              << msg;

                // cut off the exception type part
                const auto excEnd = msg.find(':');
                errorMessage = excEnd == TString::npos
                    ? msg
                    : TString{std::begin(msg) + excEnd, std::end(msg)};

                DataPuller_->OnTaskCompleted(*Module_, returnValue, execTime, errorMessage);
            }

            return;
        }

    private:
        enum class EPullTaskState {
            Waiting = 0,
            Running,
        };

        // Two tasks may run in parallel if the first one is being executed longer than a specified period
        std::atomic<EPullTaskState> State_{EPullTaskState::Waiting};
        TDefaultDataPullerPtr DataPuller_;
        IStorageConsumerProviderPtr StorageConsumerProvider_;
        IPullModulePtr Module_;
        TDuration Period_;

        ITransformingConsumerPtr TransformingConsumer_;
    };

    void Schedule(
            IPullModulePtr module,
            TDuration period,
            IStorageConsumerProviderPtr storageConsumerProvider,
            const TTransformationsConfig* transformations) override
    {
        SA_LOG(DEBUG) << "schedule pulling from " << module->Name()
                      << " every " << period;

        Y_ENSURE(period >= TDuration::Seconds(1),
                "too small pull period: " << period
                 << " for " << module->Name() << " module");

        auto timerTask = MakeFuncTimerTask(ThreadPool_, TPullTask{this, std::move(storageConsumerProvider),
                                                                  module, period, transformations});

        with_lock (PullTasksMutex_) {
            intptr_t key = reinterpret_cast<intptr_t>(module.Get());
            auto res = PullTasks_.emplace(key, timerTask);
            Y_ENSURE(res.second, "given module is already scheduled");
            RegisterName(module->Name());
        }

        auto periodMs = period.MilliSeconds();
        auto initialDelay = TDuration::MilliSeconds(periodMs - (TInstant::Now().MilliSeconds() % periodMs));

        Timer_.Schedule(timerTask, initialDelay, period);
        OnTaskChanged(*module, *timerTask);
        OnTaskScheduled(*module);
    }

    void CancelTasks() noexcept {
        SA_LOG(DEBUG) << "cancelling all pull tasks";

        with_lock(PullTasksMutex_) {
            for (auto&& [key, task]: PullTasks_) {
                task->Cancel();
            }
            PullTasks_.clear();
        }
    }

    void Cancel(const IPullModule& module) noexcept override {
        SA_LOG(DEBUG) << "cancelling " << module.Name();

        with_lock (PullTasksMutex_) {
            DeregisterName(module.Name());

            intptr_t key = reinterpret_cast<intptr_t>(&module);
            auto it = PullTasks_.find(key);
            if (it != PullTasks_.end()) {
                it->second->Cancel();
                OnTaskChanged(module, *it->second);

                PullTasks_.erase(it);
            }
        }

        TString moduleName{module.Name()};
        auto gcTask = MakeFuncTimerTask(ThreadPool_, [=] {
            if (!Status_) {
                return;
            }

            Status_->RemoveModuleData(moduleName);
        });

        Timer_.Schedule(gcTask, GcDelay_);
     }

    void OnTaskChanged(const IPullModule& module, const ITimerTask& task) noexcept {
        if (!Status_) {
            return;
        }

        Status_->UpdateModuleState(module.Name(), task.State());
    }

    void OnTaskSkipped(const IPullModule& module, TDuration waitTime = TDuration::Zero()) {
        if (!Status_) {
            return;
        }

        Status_->UpdateModuleDataOnSkip(module.Name(), waitTime);
    }

    void OnTaskScheduled(const IPullModule& module) {
        if (!Status_) {
            return;
        }

        TDuration interval;
        TTaskState::EState state;
        TString moduleName;

        with_lock (PullTasksMutex_) {
            intptr_t key = reinterpret_cast<intptr_t>(&module);
            const auto it = PullTasks_.find(key);

            Y_ENSURE(it != std::end(PullTasks_), "not even scheduled pull task is already cancelled");

            interval = it->second->Period();
            state = it->second->State();
            moduleName = module.Name();
        }
        TModuleStatus status;
        status.Name = moduleName;
        status.Interval = interval;
        status.Timestamp = TInstant::Now();
        status.SchedulerState = state;
        Status_->UpdateModuleData(std::move(status));
    }

    void OnTaskCompleted(
            const IPullModule& module,
            int returnValue,
            TDuration execTime,
            TMaybe<TString> errorMessage) noexcept
    {
        if (!Status_) {
            return;
        }

        TDuration interval;
        TTaskState::EState state;
        TString moduleName;

        with_lock (PullTasksMutex_) {
            intptr_t key = reinterpret_cast<intptr_t>(&module);
            const auto it = PullTasks_.find(key);

            if (it == std::end(PullTasks_)) {
                SA_LOG(DEBUG) << module.Name() << " completed, but has been already cancelled";
                return;
            }

            interval = it->second->Period();
            state = it->second->State();
            moduleName = module.Name();
        }
        TModuleStatus status;
        status.Name = moduleName,
        status.Interval = interval;
        status.ReturnValue = returnValue;
        status.ErrorMessage = errorMessage;
        status.Timestamp = TInstant::Now();
        status.ExecTime = execTime;
        status.SchedulerState = state;
        Status_->UpdateModuleData(std::move(status));
    }

    void RegisterName(TStringBuf name) {
        auto it = LoadedModules_.find(name);

        if (it != std::end(LoadedModules_)) {
            it->second++;
            SA_LOG(WARN) << "Module with name " << name << " is already scheduled";
        } else {
            LoadedModules_.emplace(name, 1);
        }
    }

    void DeregisterName(TStringBuf name) {
        auto it = LoadedModules_.find(name);

        if (it == std::end(LoadedModules_)) {
            SA_LOG(WARN) << "Name " << name << " is not in the LoadedModules_";
        } else if (it->second == 1) {
            LoadedModules_.erase(it);
        } else {
            it->second--;
        }
    }

private:
    TTimerThread Timer_;
    IThreadPool* ThreadPool_;
    TAdaptiveLock PullTasksMutex_;
    THashMap<intptr_t, ITimerTaskPtr> PullTasks_;
    THashMap<TString, ui32> LoadedModules_;
    TCountdownEvent RunningTasksCountdown_;

    TPullerStatus* Status_;
    TDuration GcDelay_;
};

} // namespace


IDataPullerPtr CreateDataPuller(IThreadPool* pool, TPullerStatus* status, TDuration gcDelay) {
    return new TDefaultDataPuller(pool, status, gcDelay);
}

} // namespace NAgent
} // namespace NSolomon
