#include <solomon/agent/lib/puller/data_puller.h>
#include <solomon/agent/lib/storage/sharded_storage.h>
#include <solomon/agent/lib/storage/stub/stub_storage.h>

#include <solomon/libs/cpp/threading/pool/pool.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NAgent;

///////////////////////////////////////////////////////////////////////////////
// TTestPullModule
///////////////////////////////////////////////////////////////////////////////
class TTestPullModule: public IPullModule {
public:
    TTestPullModule() {
        PullRate_ = Metrics_.Counter({ {"pulls", "count"} });
    }

    TStringBuf Name() const override {
        return "TestPullModule";
    }

    int Pull(TInstant time, IMetricConsumer* consumer) override {
        PullRate_->Inc();
        Metrics_.Accept(time, consumer);

        return 0;
    }

private:
    TMetricRegistry Metrics_;
    TCounter* PullRate_;
};

class TTestModuleWithNoTsInData: public TTestPullModule {
public:
    TTestModuleWithNoTsInData() {}

    TStringBuf Name() const override {
        return "TTestModuleWithNoTsInData";
    }

    int Pull(TInstant, IMetricConsumer* consumer) override {
        consumer->OnStreamBegin();

        {
            consumer->OnMetricBegin(NMonitoring::EMetricType::COUNTER);

            consumer->OnLabelsBegin();
            consumer->OnLabel("metric", "one");
            consumer->OnLabelsEnd();

            consumer->OnUint64(TInstant::Zero(), 123);

            consumer->OnMetricEnd();
        }

        {
            consumer->OnMetricBegin(NMonitoring::EMetricType::COUNTER);

            consumer->OnLabelsBegin();
            consumer->OnLabel("metric", "two");
            consumer->OnLabelsEnd();

            consumer->OnUint64(TInstant::Zero(), 456);

            consumer->OnMetricEnd();
        }

        consumer->OnStreamEnd();

        return 0;
    }
};

class TTestCancelPullModule: public TTestPullModule {
    int Pull(TInstant time, IMetricConsumer* consumer) override {
        if (CallCount < 2) {
            ++CallCount;
            return TTestPullModule::Pull(time, consumer);
        }

        return -1;
    }

public:
    int CallCount{0};
};

TEST(TDataPullerTest, Schedule) {
    NTest::TStubStorage storage;
    IStorageConsumerProviderPtr storageConsumerProvider = CreateStorageConsumerProviderForShard(&storage, "project", "service");

    auto poolPtr = NSolomon::CreateThreadPool(/*threads*/4, /*unlimited queue*/0);
    IDataPullerPtr puller = CreateDataPuller(poolPtr.get());
    puller->Start();

    IPullModulePtr module = new TTestPullModule;
    puller->Schedule(module, TDuration::Seconds(1), std::move(storageConsumerProvider));

    Sleep(TDuration::Seconds(3));

    puller->Stop();
    poolPtr.reset(); // Wait till every concurrent task stops

    NTest::TChunks chunks = storage.Chunks();
    NTest::TChunk& chunk = chunks[0];

    ASSERT_EQ(chunk.size(), 1u);

    NTest::TMetric& metric = chunk[0];
    ASSERT_EQ(metric.Type, EMetricType::COUNTER);
    ASSERT_EQ(metric.Points.size(), 1u);

    auto point = metric.Points[0];
    ASSERT_TRUE(point.Value >= 1 && point.Value <= 4);
    ASSERT_TRUE(point.Time > TInstant::Zero());

    ASSERT_EQ(metric.Labels.size(), 1u);
    auto label = metric.Labels[0];
    ASSERT_EQ(label.Name(), "pulls");
    ASSERT_EQ(label.Value(), "count");
}

TEST(TDataPullerTest, CancelledByRetCode) {
    NTest::TStubStorage storage;
    IStorageConsumerProviderPtr consumerProvider = CreateStorageConsumerProviderForShard(&storage, "project", "service");

    auto poolPtr = NSolomon::CreateThreadPool(/*threads*/4, /*unlimited queue*/0);
    IDataPullerPtr puller = CreateDataPuller(poolPtr.get());
    puller->Start();

    auto module = ::MakeIntrusive<TTestCancelPullModule>();
    puller->Schedule(module, TDuration::Seconds(1), std::move(consumerProvider));

    Sleep(TDuration::Seconds(4));

    puller->Stop();
    poolPtr.reset(); // Wait till every concurrent task stops

    ASSERT_EQ(module->CallCount, 2);
}

TEST(TDataPullerTest, ModuleDataWithNoTs) {
    NTest::TStubStorage storage;
    IStorageConsumerProviderPtr consumerProvider = CreateStorageConsumerProviderForShard(&storage, "project", "service");

    auto poolPtr = NSolomon::CreateThreadPool(/*threads*/4, /*unlimited queue*/0);
    IDataPullerPtr puller = CreateDataPuller(poolPtr.get());
    puller->Start();

    TDuration pullInterval = TDuration::Seconds(15);
    ui64 intervalSec = pullInterval.Seconds();
    ui64 nowSec = TInstant::Now().Seconds();
    TInstant intervalStart = TInstant::Seconds(nowSec - (nowSec % intervalSec) + intervalSec);
    TDuration delay = TDuration::Seconds(intervalSec - (nowSec % intervalSec) + 1);

    auto module = ::MakeIntrusive<TTestModuleWithNoTsInData>();
    puller->Schedule(module, pullInterval, std::move(consumerProvider));

    Sleep(delay);

    puller->Stop();
    poolPtr.reset(); // Wait till every concurrent task stops

    NTest::TChunks chunks = storage.Chunks();
    NTest::TChunk& chunk = chunks[0];

    ASSERT_EQ(chunk.size(), 2u);

    {
        NTest::TMetric& metric = chunk[0];
        ASSERT_EQ(metric.Type, EMetricType::COUNTER);
        ASSERT_EQ(metric.Points.size(), 1u);

        auto point = metric.Points[0];
        ASSERT_EQ(point.Value, 123u);
        ASSERT_EQ(point.Time, intervalStart);

        ASSERT_EQ(metric.Labels.size(), 1u);
        auto label = metric.Labels[0];
        ASSERT_EQ(label.Name(), "metric");
        ASSERT_EQ(label.Value(), "one");
    }

    {
        NTest::TMetric& metric = chunk[1];
        ASSERT_EQ(metric.Type, EMetricType::COUNTER);
        ASSERT_EQ(metric.Points.size(), 1u);

        auto point = metric.Points[0];
        ASSERT_EQ(point.Value, 456u);
        ASSERT_EQ(point.Time, intervalStart);

        ASSERT_EQ(metric.Labels.size(), 1u);
        auto label = metric.Labels[0];
        ASSERT_EQ(label.Name(), "metric");
        ASSERT_EQ(label.Value(), "two");
    }
}
