#include "pusher.h"

#include <solomon/agent/lib/auth/auth.h>
#include <solomon/agent/lib/auth/token_processor.h>
#include <solomon/agent/lib/context/context.h>
#include <solomon/agent/lib/http/headers.h>
#include <solomon/agent/lib/storage/query.h>
#include <solomon/agent/lib/storage/sequence_number.h>
#include <solomon/agent/misc/countdown_event.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/timer_thread.h>
#include <solomon/agent/protos/agent_config.pb.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/digest/md5/md5.h>
#include <library/cpp/http/misc/http_headers.h>
#include <library/cpp/string_utils/url/url.h>

#include <util/datetime/cputimer.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/generic/scope.h>
#include <util/stream/output.h>

using namespace NMonitoring;

template<>
void Out<NSolomon::NAgent::TShardKey>(IOutputStream& o, const NSolomon::NAgent::TShardKey& shardKey) {
    o << TStringBuf("Shard{project=") << shardKey.Project;
    o << TStringBuf("; cluster=") << shardKey.Cluster;
    o << TStringBuf("; service=") << shardKey.Service;
    o << TStringBuf("}");
}

namespace NSolomon::NAgent {

using TLabel = TAgentLabel;
using TLabels = TAgentLabels;

namespace {
    constexpr auto DEFAULT_INFLIGHT_LIMIT = 100;
    constexpr auto DEFAULT_REQUEST_QUEUE_SIZE_LIMIT = 0; // unlimited

    TString GetClusterValueFromContextOrConfig(const TPushConfig& pushConfig) {
        if (auto cluster = GetAgentCtx()->GetCluster()) {
            return cluster;
        }

        // TODO(ivanzhukov@): deprecate a Cluster field in the Push config
        return pushConfig.GetCluster();
    }

    struct TShardMeta {
        TAdaptiveLock StorageReadLock_;
        bool IsFirstRead_ = true;
        TSeqNo CurrOffset_;

        TShardMeta() {} // required since TAdaptiveLock() is generated protected
    };

    class TShardMetaMap {
    public:
        using TShardMetaPtr = TAtomicSharedPtr<TShardMeta>;

        TShardMeta* Get(const TShardKey& shardKey) const {
            auto g = Guard(Lock_);
            auto it = Map_.find(shardKey);

            if (it == Map_.end()) {
                return nullptr;
            }

            return it->second.Get();
        }

        void Erase(const TShardKey& shardKey) {
            auto g = Guard(Lock_);
            Map_.erase(shardKey);
        }

        template <typename... Args>
        void Insert(const TShardKey& shardKey, Args&&... args) {
            auto g = Guard(Lock_);
            if constexpr(sizeof...(args) > 0) {
                Map_.emplace(shardKey, std::forward<Args>(args)...);
            } else {
                Map_[shardKey] = ::MakeAtomicShared<TShardMeta>();
            }
        }

    private:
        THashMap<TShardKey, TShardMetaPtr, TShardKeyHasher> Map_;
        TAdaptiveLock Lock_;
    };

    TStringBuf API_V1_PATH = "/push";
    TStringBuf EMPTY_PATH = "/";

    TStringBuf GetUrlForEndpointType(
            const TEndpointWithAuth& endpoint,
            THashSet<TString>& urlsSet,
            THashSet<EClusterType>& typesSet)
    {
        if (!endpoint.HasType()) {
            Y_ENSURE(endpoint.HasUrl(), "either Type or Url should be specified inside an endpoint");

            return endpoint.GetUrl();
        }

        auto endpointType = endpoint.GetType();

        bool isTypeNew = typesSet.emplace(endpointType).second;
        Y_ENSURE(isTypeNew,
                 "The same endpoint type is specified multiple times in a Push config: "
                 << EClusterType_Name(endpointType));

        TStringBuf url;

        switch (endpointType) {
            case EClusterType::PRODUCTION:
                url = TStringBuf("https://solomon.yandex.net/api/v2/push");
                break;
            case EClusterType::PRESTABLE:
                url = TStringBuf("https://solomon-prestable.yandex.net/api/v2/push");
                break;
            case EClusterType::TESTING:
                url = TStringBuf("https://solomon-test.yandex.net/api/v2/push");
                break;
            case EClusterType::CLOUD_PROD:
                url = TStringBuf("https://solomon.cloud.yandex-team.ru/api/v2/push");
                break;
            case EClusterType::CLOUD_PREPROD:
                url = TStringBuf("https://solomon.cloud-preprod.yandex-team.ru/api/v2/push");
                break;
            case EClusterType::CLOUD_GPN:
                url = TStringBuf("https://monitoring.private-api.ycp.gpn.yandexcloud.net/api/v2/push");
                break;
            default:
                ythrow yexception() << "unsupported endpoint type: "
                                    << EClusterType_Name(endpointType);
        }

        // Check custom urls as well
        bool isUrlNew = urlsSet.emplace(url).second;
        Y_ENSURE(isUrlNew,
                 "The same url is specified multiple times in a Push config: " << url);

        return url;
    }

    class TDefaultDataPusher: public IDataPusher {
    public:
        TDefaultDataPusher(
                TSimpleSharedPtr<IThreadPool> pool,
                const TPushConfig& pushConfig,
                const TLabels& commonLabels,
                TShardedStorage* storage,
                TMetricRegistry& registry,
                TTimerThread& timer,
                TRegistrationTaskPtr registrationTask,
                IDataPusherStatusListenerPtr statusListener)
            : CommonLabels_{commonLabels}
            , Timer_{"PushScheduler"}
            , ThreadPool_{pool}
            , Storage_{storage}
            , Registry_{registry}
            , TimerThread_{timer}
            , RegistrationTask_{registrationTask}
            , AuthTokenProcessor_{TAuthTokenProcessor::Instance()}
            , StatusListener_(std::move(statusListener))
        {
            Y_ENSURE(!(pushConfig.GetHosts().empty() && pushConfig.GetEndpoints().empty()),
                    "either Hosts or Endpoints should be specified");
            Y_ENSURE(pushConfig.GetHosts().empty() ^ pushConfig.GetEndpoints().empty(),
                    "specify only one of fields: Hosts or Endpoints");

            Cluster_ = GetClusterValueFromContextOrConfig(pushConfig);
            Y_ENSURE(!Cluster_.empty(), "Empty Cluster value in the config");

            if (!pushConfig.GetHosts().empty()) {
                SA_LOG(WARN) << "field Hosts is deprecated and will be removed in later versions. Use Endpoints instead";
            }

            THashSet<TString> urlsSet;
            THashSet<EClusterType> typesSet;

            for (const THostConfig& hostConfig: pushConfig.GetHosts()) {
                const TString& host = hostConfig.GetHost();
                const ui32 port = hostConfig.GetPort();
                const TString& url = hostConfig.GetUrl();

                bool isOnlyUrlSpecified = !url.empty() && port == 0 && host.empty();
                bool isHostSpecifiedAndNotUrl = !host.empty() && url.empty();

                // TODO: deprecate a Hosts field
                Y_ENSURE(isOnlyUrlSpecified || isHostSpecifiedAndNotUrl,
                        "Specify a url in either Url or {Host, Port, Path} parameters");

                if (isHostSpecifiedAndNotUrl) {
                    Y_ENSURE(port <= Max<ui16>(), "Port value is out of range: " << port);
                }

                auto [clusterName, endpoint] = BuildEndpoint(host, port, url);

                bool isInserted = urlsSet.emplace(endpoint).second;
                Y_ENSURE(isInserted,
                        "The same url is specified multiple times in a Push config: " << url);

                ClusterNameToEndpoint_[clusterName] = endpoint;
            }

            bool isOldAuthEnabled = AuthTokenProcessor_->HasToken();

            for (const TEndpointWithAuth& endpointWithAuth: pushConfig.GetEndpoints()) {
                ui32 port = 0;
                TStringBuf url = GetUrlForEndpointType(endpointWithAuth, urlsSet, typesSet);

                auto [clusterName, normalizedUrl] = BuildEndpoint("", port, url);
                ClusterNameToEndpoint_[clusterName] = normalizedUrl;

                if (endpointWithAuth.HasIamConfig() ||
                    endpointWithAuth.HasTvmConfig() ||
                    endpointWithAuth.HasOAuthConfig() ||
                    endpointWithAuth.GetAuthMethod())
                {
                    Y_ENSURE(
                            !isOldAuthEnabled,
                            "use only one auth method: Push::Endpoints with auth or an env variable");
                }

                NSolomon::IAuthProviderPtr authProvider;
                if (const auto& authMethod = endpointWithAuth.GetAuthMethod()) {
                    authProvider = GetAgentCtx()->GetAuthProvider(authMethod);
                } else {
                    // TODO(ivanzhukov@): remove this whole else block after auth *Conf fields deprecation

                    // transform the old format to the new one for backward compatibility
                    auto newFormatAuth{endpointWithAuth};

                    auto clusterType = endpointWithAuth.GetType();
                    if (endpointWithAuth.HasTvmConfig()) {
                        newFormatAuth.MutableTvmConfig()->SetCluster(clusterType);
                    } else if (endpointWithAuth.HasIamConfig()) {
                        newFormatAuth.MutableIamConfig()->SetCluster(clusterType);
                    }

                    authProvider = CreateAuthProvider(newFormatAuth, Registry_, TimerThread_, ThreadPool_);
                }

                if (authProvider) {
                    UrlToAuthProvider_[normalizedUrl] = authProvider;
                }
            }

            if (!RegistrationTask_) { // compute this only once to prevent redundant copying
                for (const auto& [clusterName, endpoint]: ClusterNameToEndpoint_) {
                    Endpoints_.emplace_back(endpoint);
                }
            }

            const auto maxInflight = pushConfig.GetMaxInflight();
            const auto queueSizeLimit = pushConfig.GetRequestQueueSizeLimit();
            TCurlClientOptions curlOpts;
            curlOpts.DnsCacheLifetime = TDuration::Hours(2);
            curlOpts.MaxInflight = maxInflight ? maxInflight : DEFAULT_INFLIGHT_LIMIT;
            curlOpts.QueueSizeLimit = queueSizeLimit ? queueSizeLimit : DEFAULT_REQUEST_QUEUE_SIZE_LIMIT;

            TString caCertDir = Strip(GetEnv("SA_CAPATH", ""));

            if (!caCertDir.empty()) {
                curlOpts.CaCertDir = caCertDir;
            }

            HttpClient_ = CreateCurlClient(curlOpts, Registry_);

            Timer_.Start();

            TDuration pushInterval = pushConfig.GetPushInterval()
                ? TDuration::Parse(pushConfig.GetPushInterval())
                : TDuration::Seconds(15);
            TDuration retryInterval = pushConfig.GetRetryInterval()
                ? TDuration::Parse(pushConfig.GetRetryInterval())
                : TDuration::Seconds(5);

            if (retryInterval < TDuration::Seconds(5)) {
                retryInterval = TDuration::Seconds(5);

                SA_LOG(WARN) << "The minimal value for RetryInterval is \"5s\","
                             << " but \"" << pushConfig.GetRetryInterval() << "\" is given."
                             << " Falled back to 5 seconds";
            }

            ui32 retryTimes = pushConfig.GetRetryTimes()
                ? pushConfig.GetRetryTimes()
                : 3;

            bool isAllShardsParameterSpecified = pushConfig.GetAllShards();
            bool areShardsSpecifiedManually = pushConfig.ShardsSize() > 0;

            Y_ENSURE(isAllShardsParameterSpecified || areShardsSpecifiedManually,
                    "Specify shards either using AllShards: true or Shards: [...]");
            Y_ENSURE(
                !(isAllShardsParameterSpecified && areShardsSpecifiedManually),
                "Either AllShards: true or Shards: [...] should be specified in a Push config, not both"
            );

            if (pushConfig.HasShardKeyOverride()) {
                auto shardKey = pushConfig.GetShardKeyOverride();
                Y_ENSURE(!shardKey.GetProject().empty() && !shardKey.GetCluster().empty() && !shardKey.GetService().empty(),
                        "All fields of ShardKeyOverride have to be specified (Project, Cluster, Service)");

                ShardKeyOverride_ = shardKey;
            }

            if (pushConfig.ShardsSize()) {
                for (const TShardConfig& shardConfig: pushConfig.GetShards()) {
                    Y_ENSURE(shardConfig.GetProject(), "Empty project value in a Shard config");
                    Y_ENSURE(shardConfig.GetService(), "Empty service value in a Shard config");

                    Schedule(
                        shardConfig.GetProject(), shardConfig.GetService(),
                        pushInterval, retryInterval,
                        retryTimes
                    );
                }
            } else {
                ScheduleAllShards(pushInterval, retryInterval, retryTimes);
            }

            SA_LOG(INFO) << "pusher is initialized";
            if (ShardKeyOverride_) {
                SA_LOG(INFO) << "overriding shard keys with:"
                             << " project=" << ShardKeyOverride_->GetProject()
                             << ", cluster=" << ShardKeyOverride_->GetCluster()
                             << ", service=" << ShardKeyOverride_->GetService();
            }
        }

        ~TDefaultDataPusher() {
            try {
                Stop();
            } catch (...) {}
        }

    private:
        void Start() override {
            auto g = Guard(PushTasksLock_);

            for (const auto& it: PushTasks_) {
                const TShardKey& shardKey = it.first;
                auto [task, interval] = it.second;

                if (task->State() == TTaskState::VIRGIN) {
                    Timer_.Schedule(task, Jitter_(interval), interval);
                    SA_LOG(INFO) << "schedule a Push task for " << shardKey << " with " << interval << " interval";
                }
            }
        }

        void Stop() override {
            Timer_.Stop();
            RunningTasksCountdown_.Stop();

            while (!RunningTasksCountdown_.Await(TDuration::Seconds(10))) {
                SA_LOG(DEBUG) << "waiting for request tasks completion";
            }
        }

        TString ConstructQueryString(const TShardKey& shardKey) {
            TCgiParameters queryString;
            queryString.InsertUnescaped(NLabels::LABEL_PROJECT, shardKey.Project);
            queryString.InsertUnescaped(NLabels::LABEL_CLUSTER, shardKey.Cluster);
            queryString.InsertUnescaped(NLabels::LABEL_SERVICE, shardKey.Service);

            return TStringBuilder() << queryString.Print();
        }

        const TString PUSHER_PREFIX = "PUSHER";
        TString ConstructConsumerId(const TShardKey& shardKey) {
            return PUSHER_PREFIX + ":" + shardKey.Project + ":" + shardKey.Service;
        }

        TMaybe<TSeqNo> ReadDataFromStorage(const TShardKey& shardKey, NMonitoring::IMetricEncoder* encoder) {
            try {
                // It'd be more efficient to create a single stream and a single encoder
                // and to share them in every request, but tasks can overlap (if one
                // is running longer than an update interval) -> objects will be used in parallel,
                // which is not desirable

                auto* shardMeta = ShardsMeta_.Get(shardKey);
                if (shardMeta == nullptr) {
                    return Nothing();
                }

                TReadResult result;
                with_lock (shardMeta->StorageReadLock_) {
                    TQuery query;
                    if (Y_LIKELY(!shardMeta->IsFirstRead_)) {
                        // Note: if no offset is specified, then a storage will get the commited offset for a
                        // specific consumerId. Hence, not specifying it is the same as specifying the last
                        // commited value. Will be useful after a restart with a persistent storage.
                        query.Offset(shardMeta->CurrOffset_);
                        Storage_->Commit(shardKey.Project, shardKey.Service, ConstructConsumerId(shardKey),
                                         shardMeta->CurrOffset_);
                    }

                    result = Storage_->Read(shardKey.Project, shardKey.Service, query, encoder);

                    shardMeta->IsFirstRead_ = false;
                    shardMeta->CurrOffset_ = result.SeqNo;

                    if (result.NumOfMetrics == 0) {
                        // No new data were available
                        return Nothing();
                    }
                }

                if (!ShardKeyOverride_ && !CommonLabels_.Empty()) {
                    encoder->OnLabelsBegin();
                    for (auto&& l: CommonLabels_) {
                        encoder->OnLabel(TString{l.Name()}, TString{l.Value()});
                    }
                    encoder->OnLabelsEnd();
                }

                encoder->Close();

                // TODO (SOLOMON-3560): Support SeqNo in a persistent storage
                return result.SeqNo;
            } catch (...) {
                SA_LOG(ERROR) << shardKey << " failed to read data from the storage: "
                              << CurrentExceptionMessage();
                return Nothing();
            }
        }

        TString ConstructRequestId(const TString& data, const TSeqNo& seqNo, const TShardKey& shardKey) {
            MD5 r;
            r.Update(data);
            r.Update(reinterpret_cast<const char*>(&seqNo), sizeof(seqNo));
            // Additional salt
            r.Update(shardKey.Project);
            r.Update(shardKey.Cluster);
            r.Update(shardKey.Service);

            return r.End(nullptr);
        }

        void SchedulePostRequest(
                TStringBuf hostUrl,
                const TString& queryStringBase,
                const TString requestId,
                const TShardKey& shardKey,
                const TString& dataToSend,
                TDuration retryInterval,
                ui32 retryTimes)
        {
            TString logPrefix = TStringBuilder() << shardKey << " [rid=" << requestId << "]";

            SA_LOG(DEBUG) << logPrefix << " sending data to " << hostUrl;

            TRequestOpts opts;
            opts.ConnectTimeout = TDuration::Seconds(5);
            opts.ReadTimeout = TDuration::Minutes(5);
            opts.Retries = ui8(retryTimes);
            opts.BackoffMin = retryInterval;
            opts.BackoffMax = retryInterval * 2;

            auto headers = Headers({
                {TString{NHttpHeaders::CONTENT_TYPE}, "application/x-solomon-spack"},
                {TString{NHttpHeaders::USER_AGENT}, USER_AGENT_HEADER},
                {TString{"X-Solomon-Project"}, shardKey.Project},
                {TString{"X-Solomon-Cluster"}, shardKey.Cluster},
                {TString{"X-Solomon-Service"}, shardKey.Service},
            });

            if (auto it = UrlToAuthProvider_.find(hostUrl); it != UrlToAuthProvider_.end()) {
                THashMap<TString, TString> tmpHeaders;
                it->second->AddCredentials(tmpHeaders);

                for (auto& [hName, hValue]: tmpHeaders) {
                    headers->Add(hName, hValue);
                }
            } else {
                AuthTokenProcessor_->InflateHttpHeaders(*headers);
            }

            TString url = TStringBuilder() << hostUrl << (hostUrl.Contains('?') ? '&' : '?')
                                           << queryStringBase << "&requestId=" << requestId;
            // XXX: reuse headers?
            auto req = Post(std::move(url), dataToSend, std::move(headers));

            HttpClient_->Request(std::move(req), [=, host{TString{hostUrl}}, timer{TSimpleTimer()}](IHttpClient::TResult result) {
                    Y_SCOPE_EXIT(this) {
                        RunningTasksCountdown_.Dec();
                    };

                    TDuration respTime = timer.Get();
                    bool failed = false;

                    if (!result.Success()) {
                        failed = true;
                        SA_LOG(WARN) << logPrefix << " failed to send data to " << host << ": " << result.Error().Message();
                    } else if (result.Value()->Code() >= 400) {
                        failed = true;
                        SA_LOG(WARN) << logPrefix << " failed to send data to " << host << " with code " << result.Value()->Code() << ": " << result.Value()->Data();
                    } else {
                        SA_LOG(DEBUG) << logPrefix << " successfully sent data to " << host;
                    }

                    OnRequestCompleted(shardKey, host, failed, respTime);
            }, opts);

            OnRequestStarted(shardKey, TString{hostUrl});
        }

        void SendDataForOneShard(
                const TVector<TStringBuf>& hostsUrls,
                const TShardKey& shardKey,
                TDuration retryInterval,
                ui32 retryTimes,
                const TString& dataForSending,
                TSeqNo seqNo)
        {
            ui64 numOfRequests = hostsUrls.size();
            TString requestId;
            try {
                requestId = ConstructRequestId(dataForSending, seqNo, shardKey);
            } catch (...) {
                SA_LOG(ERROR) << shardKey << " failed to construct a requestId: "
                              << CurrentExceptionMessage();

                RunningTasksCountdown_.Sub(numOfRequests);
                return;
            }

            const TString queryStringBase = ConstructQueryString(shardKey);

            for (size_t i = 0; i != numOfRequests; ++i) {
                SchedulePostRequest(
                    hostsUrls[i], queryStringBase, requestId,
                    shardKey,
                    dataForSending,
                    retryInterval, retryTimes);
            }
        }

        bool Schedule(
                const TString& project,
                const TString& service,
                TDuration pushInterval,
                TDuration retryInterval,
                ui32 retryTimes,
                bool instantRun = false)
        {
            auto g = Guard(PushTasksLock_);

            TShardKey shardKey{project, Cluster_, service};

            auto findIt = PushTasks_.find(shardKey);
            if (findIt != PushTasks_.end()) {
                return false;
            }

            ShardsMeta_.Insert(shardKey);

            TShardKeySubstitutions shardKeySubstitutions;
            if (ShardKeyOverride_) {
                shardKeySubstitutions = TShardKeySubstitutions{
                   ShardKeyOverride_->GetProject(),
                   ShardKeyOverride_->GetCluster(),
                   ShardKeyOverride_->GetService(),
                   ShardKeyOverride_->GetDoNotAppendHostLabel(),
                };
            }

            auto task = MakeFuncTimerTask(
                ThreadPool_.Get(),
                [=]() -> void {
                    // TODO(SOLOMON-4909): Do not start a push task if a previous one is still in progress

                    if (!RunningTasksCountdown_.TryAdd(1)) {
                        // Counter is already stopped
                        return;
                    }

                    TVector<TStringBuf> endpointsLeft;

                    if (RegistrationTask_) {
                        for (const auto& [clusterName, endpoint]: ClusterNameToEndpoint_) {
                            // TODO: a place for an optimization: if a registration request is being retried,
                            // Pusher will wait the lock and other tasks can be started if we wait long enough
                            if (RegistrationTask_->IsRegisteredIn(clusterName)) {
                                SA_LOG(DEBUG) << shardKey << " skipping a push task for " << clusterName
                                              << ", because it's being handled by multishard pulling";
                            } else {
                                endpointsLeft.emplace_back(endpoint);
                            }
                        }
                    }

                    const TVector<TStringBuf>& endpoints = RegistrationTask_ ? endpointsLeft : Endpoints_;

                    if (endpoints.empty()) {
                        SA_LOG(DEBUG) << shardKey << " skipping a push task, because there are no consumers left";
                        return;
                    }

                    if (shardKeySubstitutions) {
                        TGroupingEncoder proxyEncoder(shardKeySubstitutions, SpackEncoderFactory);
                        TMaybe<TSeqNo> seqNo = ReadDataFromStorage(shardKey, &proxyEncoder);

                        if (!seqNo) {
                            SA_LOG(DEBUG) << shardKey << " no data to send, not pushing";

                            RunningTasksCountdown_.Sub(1);
                            return;
                        }

                        TShardsData shardsData = proxyEncoder.GetShardsData();

                        Y_ENSURE(RunningTasksCountdown_.TryAdd(shardsData.size() * endpoints.size()));
                        RunningTasksCountdown_.Sub(1);

                        for (auto&& it: shardsData) {
                            const TString& data = it.second;
                            TShardKey newShardKey = TShardKey(it.first.Project, it.first.Cluster, it.first.Service);

                            SendDataForOneShard(endpoints, newShardKey, retryInterval, retryTimes, data, *seqNo);
                        }
                    } else {
                        TStringStream stringStream;
                        NMonitoring::IMetricEncoderPtr spackEncoder = NMonitoring::EncoderSpackV1(
                            &stringStream,
                            NMonitoring::ETimePrecision::SECONDS,
                            NMonitoring::ECompression::ZSTD,
                            NMonitoring::EMetricsMergingMode::MERGE_METRICS
                        );

                        TMaybe<TSeqNo> seqNo = ReadDataFromStorage(shardKey, spackEncoder.Get());
                        if (!seqNo) {
                            SA_LOG(DEBUG) << shardKey << " no data to send, not pushing";

                            RunningTasksCountdown_.Sub(1);
                            return;
                        }

                        Y_ENSURE(RunningTasksCountdown_.TryAdd(endpoints.size()));
                        RunningTasksCountdown_.Sub(1);

                        const TString& data = stringStream.Str();
                        SendDataForOneShard(endpoints, shardKey, retryInterval, retryTimes, data, *seqNo);
                    }
                });

            TTaskInfo taskInfo = { task, pushInterval };
            auto [insertIt, isInserted] = PushTasks_.emplace(shardKey, taskInfo);
            if (!isInserted) {
                return false;
            }

            if (instantRun) {
                Timer_.Schedule(task, Jitter_(pushInterval), pushInterval);
                SA_LOG(INFO) << "schedule a Push task for " << shardKey << " with " << pushInterval << " interval";
            }

            return true;
        }

        void Cancel(const TString& project, const TString& service) noexcept override {
            TShardKey shardKey{project, Cluster_, service};
            SA_LOG(DEBUG) << "cancelling a task for " << shardKey;

            with_lock (PushTasksLock_) {
                if (auto it = PushTasks_.find(shardKey); it != PushTasks_.end()) {
                    it->second.Task->Cancel();
                    PushTasks_.erase(it);
                }
            }

            ShardsMeta_.Erase(shardKey);
        }

        void ScheduleAllShards(TDuration pushInterval, TDuration retryInterval, const ui32 retryTimes) {
            auto task = MakeFuncTimerTask(ThreadPool_.Get(), [=, this_{IDataPusherPtr(this)}]() {
                Storage_->ForEachShard([=, this_{this_}](const TString& project, const TString& service) {
                    Schedule(project, service, pushInterval, retryInterval, retryTimes, true);
                });
            });

            Timer_.Schedule(task, TDuration::Zero(), TDuration::Seconds(1));
        }

        void OnRequestStarted(const TShardKey& shardKey, TString hostUrl) {
            if (StatusListener_) {
                StatusListener_->OnRequestStarted(shardKey, hostUrl);
            }
        }

        void OnRequestCompleted(const TShardKey& shardKey, TString hostUrl, int code, TDuration respTime) {
            if (StatusListener_) {
                StatusListener_->OnRequestCompleted(shardKey, hostUrl, code, respTime);
            }
        }

    private:
        const TLabels CommonLabels_;
        TString Cluster_;

        TTimerThread Timer_;
        TSimpleSharedPtr<IThreadPool> ThreadPool_;
        THalfJitter Jitter_;

        TShardedStorage* Storage_;

        THashMap<TString, TString> ClusterNameToEndpoint_;
        TVector<TStringBuf> Endpoints_;
        IHttpClientPtr HttpClient_;

        struct TTaskInfo {
            ITimerTaskPtr Task;
            TDuration Interval;
        };

        TCountdownEvent RunningTasksCountdown_;
        THashMap<TShardKey, TTaskInfo, TShardKeyHasher> PushTasks_;
        TAdaptiveLock PushTasksLock_;

        TShardMetaMap ShardsMeta_;
        TMaybe<TShardKeyOverride> ShardKeyOverride_;

        TMetricRegistry& Registry_;
        TTimerThread& TimerThread_;
        TRegistrationTaskPtr RegistrationTask_;
        TAuthTokenProcessor* AuthTokenProcessor_;
        THashMap<TString, IAuthProviderPtr> UrlToAuthProvider_;
        IDataPusherStatusListenerPtr StatusListener_;
    };

} // namespace

IDataPusherPtr CreateDataPusher(
    TSimpleSharedPtr<IThreadPool> pool,
    const TPushConfig& pushConfig,
    const TLabels& commonLabels,
    TShardedStorage* storage,
    TMetricRegistry& registry,
    TTimerThread& timer,
    TRegistrationTaskPtr registrationTask,
    IDataPusherStatusListenerPtr statusListener
) {
    return {
        new TDefaultDataPusher{pool, pushConfig, commonLabels, storage, registry, timer, registrationTask, statusListener}
    };
}

TEndpointInfo BuildEndpoint(TStringBuf configHost, ui16 configPort, TStringBuf configUrl) {
    ui16 port{0};
    TStringBuf scheme, host, path;

    if (!configUrl.empty()) {
        // to gather the path value
        SplitUrlToHostAndPath(configUrl, host, path);
    } else {
        SplitUrlToHostAndPath(configHost, host, path);
        Y_ENSURE(path.empty(), "If you want to specify a url path, use Url parameter instead of Host");

        path = API_V1_PATH;
    }

    if (!path) {
        path = EMPTY_PATH;
    }

    TStringBuf url = !configUrl.empty() ? configUrl : configHost;
    // returns default port if the scheme is present
    Y_ENSURE(TryGetSchemeHostAndPort(url, scheme, host, port), "Unable to parse " << url << " as a valid host");
    Y_ENSURE(!(port != 0 && configPort != 0), "Port values are in both Host and Port parameters. Choose only one");

    if (configPort != 0) {
        port = configPort;
    } else if (port == 0) {
        // use HTTP if set neither in Host, nor in Port
        port = 80;
    }

    if (!configUrl.empty()) {
        Y_ENSURE(scheme == "http://" || scheme == "https://",
                 "Wrong or no scheme is specified in the url \"" << url << "\". Choose http:// or https://");
    }

    if (scheme.empty()) {
        scheme = port == 443
                 ? TStringBuf("https://")
                 : TStringBuf("http://");
    }

    return { TString{host}, TStringBuilder() << scheme << host << ":" << port << path };
}

} // namespace NSolomon::NAgent
