#include <solomon/agent/lib/pusher/pusher.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/env.h>

#include <google/protobuf/text_format.h>

using namespace NSolomon;
using namespace NAgent;

TEST(TPusherTest, PortGreaterThanUint16) {
    TString host = "example.com:100000";
    TString url;
    ui16 port{0};

    ASSERT_THROW(BuildEndpoint(host, port, url), yexception);
}

TEST(TPusherTest, DefaultValues) {
    TString host = "example.com"; // path is not specified
    TString url;
    ui16 port{0}; // not specified

    ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:80/push");
}

TEST(TPusherTest, SchemeForDifferentPorts) {
    TString host = "example.com";
    TString url;

    {
        ui16 port{0};
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:80/push");
    }

    {
        ui16 port{123};
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:123/push");
    }

    {
        ui16 port{442};
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:442/push");
    }

    {
        ui16 port{443};
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "https://example.com:443/push");
    }
}

TEST(TPusherTest, PortSpecifiedInBothHostAndPort) {
    TString host = "example.com:123";
    TString url;
    ui16 port{456};

    ASSERT_THROW(BuildEndpoint(host, port, url), yexception);
}

TEST(TPusherTest, PortValueFromHostParameter) {
    TString host = "example.com:456";
    TString url;
    ui16 port{0};

    ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:456/push");
}

TEST(TPusherTest, PathIsNotAllowed) {
    TString host = "example.com/super-path";
    TString url;
    ui16 port{0};

    ASSERT_THROW(BuildEndpoint(host, port, url), yexception);
}

TEST(TPusherTest, UrlWithoutScheme) {
    TString host;
    TString url = "example.com:456/api/v2/push";
    ui16 port{0};

    ASSERT_THROW(BuildEndpoint(host, port, url), yexception);
}

TEST(TPusherTest, CorrectUrls) {
    TString host;
    ui16 port{0};

    {
        TString url = "http://example.com";
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:80/");
    }

    {
        TString url = "https://example.com";
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "https://example.com:443/");
    }

    {
        TString url = "http://example.com:456/123";
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "http://example.com:456/123");
    }

    {
        TString url = "https://example.com:456/123";
        ASSERT_EQ(BuildEndpoint(host, port, url).Endpoint, "https://example.com:456/123");
    }
}

TEST(TPusherTest, UrlWrongScheme) {
    TString host;
    TString url = "ftp://example.com";
    ui16 port{0};

    ASSERT_THROW(BuildEndpoint(host, port, url), yexception);
}

class TConfigTest: public ::testing::Test {
public:
    IDataPusherPtr CreatePusher(const TPushConfig& pushConfig) {
        return CreateDataPusher(Pool_, pushConfig, {}, nullptr, Metrics_, TimerThread_, nullptr, nullptr);
    }

    TPushConfig ParsePushConfig(const TString& str) {
        TPushConfig pushConfig;
        Y_ENSURE(google::protobuf::TextFormat::ParseFromString(str, &pushConfig), "failed to parse");

        return pushConfig;
    }

private:
    TSimpleSharedPtr<IThreadPool> Pool_ = MakeSimpleShared<TThreadPool>();
    TTimerThread TimerThread_;
    NMonitoring::TMetricRegistry Metrics_;
};

TEST_F(TConfigTest, AuthMethodsClash) {
    auto pushConfig = ParsePushConfig(R"(
        Endpoints: [ {
            Url: "http://localhost:6666"

            OAuthConfig {
                SecretFile: "oauth.secret"
            }
        } ]

        AllShards: true
        Cluster: "test"
    )");

    SetEnv("SA_AUTH_TYPE", "OAuth");
    SetEnv("SA_AUTH_TOKEN", "token_value");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR(
            CreatePusher(pushConfig),
            yexception,
            "use only one auth method");

    SetEnv("SA_AUTH_TYPE", "");
    SetEnv("SA_AUTH_TOKEN", "");
}

TEST_F(TConfigTest, HostsOrEndpoints) {
    {
        auto pushConfig = ParsePushConfig(R"(
            AllShards: true
            Cluster: "test"
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreatePusher(pushConfig),
                yexception,
                "either Hosts or Endpoints should be specified");
    }

    {
        auto pushConfig = ParsePushConfig(R"(
            Hosts: [ { Url: "http://localhost:6666" } ]
            Endpoints: [ { Url: "http://localhost:6666" } ]

            AllShards: true
            Cluster: "test"
        )");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreatePusher(pushConfig),
                yexception,
                "specify only one of fields: Hosts or Endpoints");
    }
}

TEST_F(TConfigTest, WrongEndpointType) {
    auto pushConfig = ParsePushConfig(R"(
            Endpoints: [ { Type: UNKNOWN_CLUSTER } ]

            AllShards: true
            Cluster: "test"
        )");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR(
            CreatePusher(pushConfig),
            yexception,
            "unsupported endpoint type");
}
