#include "cast_proto.h"

#include <google/protobuf/map_entry.h>

#define SET_PROTO_VALUE_CASE(ProtoTypeEnum, CppType, Value) \
    case ::google::protobuf::FieldDescriptor::ProtoTypeEnum: \
        if (field->is_repeated()) { \
            if (PyList_Check(Value)) { \
                PyListForEach(Value, [this, field](const PyObject* item) { \
                    Convert##CppType<true>(field, item); \
                }); \
            } else { \
                Convert##CppType<true>(field, Value); \
            } \
        } else { \
            Convert##CppType<false>(field, Value); \
        } \
        break


namespace NPython2 {
namespace {

template <typename TFunctor>
void PyListForEach(const PyObject* list, TFunctor fn) {
    Py_ssize_t size = PyList_GET_SIZE(list);
    for (Py_ssize_t i = 0; i < size; ++i) {
        fn(PyList_GET_ITEM(list, i));
    }
}

template <typename TFunctor>
void PyDictForEach(const PyObject* dict, TFunctor fn) {
    Py_ssize_t pos = 0;
    PyObject *namePy, *valuePy;

    auto nco = const_cast<PyObject*>(dict); // non const object
    while (PyDict_Next(nco, &pos, &namePy, &valuePy)) {
        PY_ENSURE_TYPE(PyString_Check, namePy, "dict key must be a string");
        TStringBuf name = NPython2::ToStringBufNoAlloc(namePy);

        PY_ENSURE_TYPE(PyString_Check, valuePy, "label value must be a string");
        TStringBuf value = NPython2::ToStringBufNoAlloc(valuePy);

        fn(name, value);
    }
}

} // namespace

void TProtoConverter::Convert(const PyObject* pyObj) {
    PY_ENSURE_TYPE(PyDict_Check, pyObj,
            "expected to get dict to conver to proto message "
            << Descr_->full_name());

    for (int i = 0; i < Descr_->field_count(); ++i) {
        const FieldDescriptor* field = Descr_->field(i);

        // borrowed reference
        PyObject* value = PyDict_GetItemString(
                    const_cast<PyObject*>(pyObj),
                    field->name().c_str());
        if (value == nullptr) {
            continue;
        }

        switch (field->cpp_type()) {
        SET_PROTO_VALUE_CASE(CPPTYPE_INT32, Int32, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_INT64, Int64, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_UINT32, UInt32, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_UINT64, UInt64, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_FLOAT, Float, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_DOUBLE, Double, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_BOOL, Bool, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_ENUM, Enum, value);
        SET_PROTO_VALUE_CASE(CPPTYPE_STRING, String, value);

        case FieldDescriptor::CPPTYPE_MESSAGE:
            if (field->is_map()) {
                // XXX: only map<string, string> is supported

                PY_ENSURE_TYPE(PyDict_Check, value,
                        "expected to get dict to convert to proto map field "
                        << field->full_name());

                PyDictForEach(value, [this, field](TStringBuf name, TStringBuf value) {
                    ConvertMapEntry(field, name, value);
                });
            } else if (field->is_repeated()) {
                if (PyList_Check(value)) {
                    PyListForEach(value, [this, field](const PyObject* item) {
                        ConvertMsg<true>(field, item);
                    });
                } else {
                    ConvertMsg<true>(field, value);
                }
            } else {
                ConvertMsg<false>(field, value);
            }
            break;

        default:
            PY_FAIL("unsupported proto field type: " << field->type_name());
        }
    }
}

void TProtoConverter::ConvertMapEntry(
        const FieldDescriptor* field,
        TStringBuf name, TStringBuf value)
{
    google::protobuf::Message* entry = Reflection_->AddMessage(Message_, field);
    auto* d = entry->GetDescriptor();
    auto* r = entry->GetReflection();

    {
        auto* keyField = d->field(0);
        Y_VERIFY_DEBUG(keyField->name() == "key");
        Y_VERIFY_DEBUG(keyField->type() == FieldDescriptor::TYPE_STRING);

        r->SetString(entry, keyField, TString(name));
    }
    {
        auto* valueField = d->field(1);
        Y_VERIFY_DEBUG(valueField->name() == "value");
        Y_VERIFY_DEBUG(valueField->type() == FieldDescriptor::TYPE_STRING);

        r->SetString(entry, valueField, TString(value));
    }
}

} // namespace NPython2
