#pragma once

#if !defined(INCLUDE_CAST_PROTO_INL_H)
#error "you should never include cast_proto_inl.h directly"
#endif  // INCLUDE_CAST_PROTO_INL_H


namespace NPython2 {

template <bool Repeated>
void TProtoConverter::ConvertInt32(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyInt_Check, value,
        "expected to get long to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto i32Value = PyInt_AsLong(const_cast<PyObject*>(value));
    if (Repeated) {
        Reflection_->AddInt32(Message_, field, i32Value);
    } else {
        Reflection_->SetInt32(Message_, field, i32Value);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertInt64(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyLong_Check, value,
        "expected to get long to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto i64Value = static_cast<i64>(PyLong_AsUnsignedLongLongMask(
                                         const_cast<PyObject*>(value)));
    if (Repeated) {
        Reflection_->AddInt64(Message_, field, i64Value);
    } else {
        Reflection_->SetInt64(Message_, field, i64Value);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertUInt32(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyInt_Check, value,
        "expected to get long to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto ui32Value = PyInt_AsUnsignedLongMask(const_cast<PyObject*>(value));
    if (Repeated) {
        Reflection_->AddUInt32(Message_, field, ui32Value);
    } else {
        Reflection_->SetUInt32(Message_, field, ui32Value);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertUInt64(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyLong_Check, value,
        "expected to get long to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto ui64Value = PyLong_AsUnsignedLongLongMask(const_cast<PyObject*>(value));
    if (Repeated) {
        Reflection_->AddUInt64(Message_, field, ui64Value);
    } else {
        Reflection_->SetUInt64(Message_, field, ui64Value);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertFloat(const FieldDescriptor* field, const PyObject* value) {
    // TODO: support Int conversion
    PY_ENSURE_TYPE(PyFloat_Check, value,
        "expected to get float to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto floatValue = static_cast<float>(PyFloat_AsDouble(const_cast<PyObject*>(value)));
    if (Repeated) {
        Reflection_->AddFloat(Message_, field, floatValue);
    } else {
        Reflection_->SetFloat(Message_, field, floatValue);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertDouble(const FieldDescriptor* field, const PyObject* value) {
    // TODO: support Int and Long conversion
    PY_ENSURE_TYPE(PyFloat_Check, value,
        "expected to get float to set value of "
        << Descr_->full_name() << "::" << field->name());
    auto doubleValue = PyFloat_AsDouble(const_cast<PyObject*>(value));
    if (Repeated) {
        Reflection_->AddDouble(Message_, field, doubleValue);
    } else {
        Reflection_->SetDouble(Message_, field, doubleValue);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertEnum(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyString_Check, value,
        "expected to get string to set value of "
        << Descr_->full_name() << "::" << field->name());

    TString strValue(NPython2::ToStringBufNoAlloc(value));
    const auto* enumDescr = field->enum_type();
    auto* enumValue = enumDescr->FindValueByName(strValue);
    PY_ENSURE(enumValue,
            "unknown enum value '" << strValue
            << "' of " << enumDescr->full_name());

    if (Repeated) {
        Reflection_->AddEnum(Message_, field, enumValue);
    } else {
        Reflection_->SetEnum(Message_, field, enumValue);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertBool(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyBool_Check, value,
        "expected to get bool to set value of "
        << Descr_->full_name() << "::" << field->name());
    if (Repeated) {
        Reflection_->AddBool(Message_, field, value == Py_True);
    } else {
        Reflection_->SetBool(Message_, field, value == Py_True);
    }
}

template <bool Repeated>
void TProtoConverter::ConvertString(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyString_Check, value,
        "expected to get string to set value of "
        << Descr_->full_name() << "::" << field->name());
    TStringBuf str = NPython2::ToStringBufNoAlloc(value);
    if (Repeated) {
        Reflection_->AddString(Message_, field, TString(str));
    } else {
        Reflection_->SetString(Message_, field, TString(str));
    }
}

template <bool Repeated>
void TProtoConverter::ConvertMsg(const FieldDescriptor* field, const PyObject* value) {
    PY_ENSURE_TYPE(PyDict_Check, value,
        "expected to get dict to set value of "
        << Descr_->full_name() << "::" << field->name());
    google::protobuf::Message* message = Repeated
        ? Reflection_->AddMessage(Message_, field)
        : Reflection_->MutableMessage(Message_, field);
    TProtoConverter converter(message);
    converter.Convert(value);
}

} // namespace NPython2
