#include "error.h"
#include "ptr.h"
#include "gil.h"

#include <Python.h>

namespace NPython2 {

// this function does the same as this python code:
//
//      import sys
//      import traceback
//
//      "".join(traceback.format_exception(
//              sys.last_type,
//              sys.last_value,
//              sys.last_traceback))
//
TString LastErrorAsString() {
    TGilGuard gil;

    PyObject* type = nullptr;
    PyObject* value = nullptr;
    PyObject* traceback = nullptr;

    // save the current exception
    PyErr_Fetch(&type, &value, &traceback);
    PyErr_NormalizeException(&type, &value, &traceback);
    PyErr_Clear();

    if (type == nullptr) {
        return {};
    }

    TObjectPtr typePtr(type, TObjectPtr::ADD_REF);
    TObjectPtr valuePtr(value, TObjectPtr::ADD_REF);
    TObjectPtr tbPtr(traceback, TObjectPtr::ADD_REF);
    Y_UNUSED(tbPtr);

    if (!traceback) {
        traceback = Py_None;
    }

    TObjectPtr mod(PyImport_ImportModule("traceback"));
    if (!mod) {
        return {};
    }

    TObjectPtr traceList = PyObject_CallMethod(
            mod.Get(), (char*)"format_exception", (char*)"NNN",
            type, valuePtr.Get(), traceback);

    Y_ENSURE(traceList.Get() != nullptr, "Unable to get the Python trace");

    TObjectPtr string(PyString_FromString(""));
    TObjectPtr ret(_PyString_Join(string.Get(), traceList.Get()));

    Y_ENSURE(ret.Get() != nullptr, "_PyString_Join failed");

    char* result;
    Py_ssize_t len;
    if (PyString_AsStringAndSize(ret.Get(), &result, &len) < 0) {
        return {};
    }

    return TString(result, len);
}

} // namespace NPython2
