#pragma once

#include <util/system/types.h>
#include <util/system/yassert.h>

#include <Python.h>

namespace NPython2 {
namespace NPrivate {

///////////////////////////////////////////////////////////////////////////////
// TRefCountedPtr
///////////////////////////////////////////////////////////////////////////////
template <typename T, typename Ops>
class TRefCountedPtr {
public:
    enum AddRef { ADD_REF };
    enum StealRef { STEAL_REF };

public:
    inline TRefCountedPtr(T* ptr = nullptr)
        : Ptr_(ptr)
    {
        Ref();
    }

    inline TRefCountedPtr(T* ptr, StealRef)
        : Ptr_(ptr)
    {
        // do not call Ref() on new pointer
    }

    inline TRefCountedPtr(const TRefCountedPtr& rhs)
        : Ptr_(rhs.Ptr_)
    {
        Ref();
    }

    inline TRefCountedPtr(TRefCountedPtr&& rhs)
        : Ptr_(rhs.Ptr_)
    {
        rhs.Ptr_ = nullptr;
    }

    inline TRefCountedPtr& operator=(const TRefCountedPtr& rhs) {
        if (this != &rhs) {
            UnRef();
            Ptr_ = rhs.Ptr_;
            Ref();
        }

        return *this;
    }

    inline TRefCountedPtr& operator=(TRefCountedPtr&& rhs) {
        if (this != &rhs) {
            UnRef();
            Ptr_ = rhs.Ptr_;
            rhs.Ptr_ = nullptr;
        }

        return *this;
    }

    inline ~TRefCountedPtr() {
        UnRef();
    }

    inline void Reset(T* ptr = nullptr) {
        if (Ptr_ != ptr) {
            UnRef();
            Ptr_ = ptr;
            Ref();
        }
    }

    inline void Reset(T* ptr, StealRef) {
        if (Ptr_ != ptr) {
            UnRef();
            Ptr_ = ptr;
        }
    }

    inline void Swap(TRefCountedPtr& rhs) {
        T* tmp = Ptr_;
        Ptr_ = rhs.Ptr_;
        rhs.Ptr_ = tmp;
    }

    inline T* Release() {
        // do not decrement ref counter here. just send ownership to caller
        T* tmp = Ptr_;
        Ptr_ = nullptr;
        return tmp;
    }

    inline T* Get() const { return Ptr_; }
    inline T& operator*() const { return *Ptr_; }
    inline T* operator->() const { return Ptr_; }
    inline explicit operator bool() const { return Ptr_ != nullptr; }

    inline ui32 RefCount() const {
        return Ptr_ ? Ops::RefCount(Ptr_) : 0;
    }

private:
    inline void Ref() {
        if (Ptr_){
            Ops::Ref(Ptr_);
        }
    }

    inline void UnRef() {
        if (Ptr_) {
            Ops::UnRef(Ptr_);
            Ptr_ = nullptr;
        }
    }

private:
    T* Ptr_;
};

template <typename T>
class TPyPtrOps {
public:
    static inline void Ref(T* t) {
        Y_ASSERT(t);
        Py_INCREF(t);
    }

    static inline void UnRef(T* t) {
        Y_ASSERT(t);
        Py_DECREF(t);
    }

    static inline ui32 RefCount(const T* t) {
        Y_ASSERT(t);
        return t->ob_refcnt;
    }
};

} // namespace NPrivate

///////////////////////////////////////////////////////////////////////////////
// TPyPtr
///////////////////////////////////////////////////////////////////////////////
template <typename T>
class TPyBasePtr: public NPrivate::TRefCountedPtr<T, NPrivate::TPyPtrOps<T>> {
    using TSelf = NPrivate::TRefCountedPtr<T, NPrivate::TPyPtrOps<T>>;

public:
    inline TPyBasePtr()
    {
    }

    inline TPyBasePtr(T* p)
        : TSelf(p, TSelf::STEAL_REF)   // do not increment refcounter by default
    {
    }

    inline TPyBasePtr(T* p, typename TSelf::AddRef)
        : TSelf(p)
    {
    }

    inline void ResetSteal(T* p) {
        TSelf::Reset(p, TSelf::STEAL_REF);
    }

    inline void ResetAddRef(T* p) {
        TSelf::Reset(p);
    }

    inline void Reset() {
        TSelf::Reset();
    }

    void Reset(T* p) = delete;
};

using TObjectPtr = TPyBasePtr<PyObject>;
using TCodeObjectPtr = TPyBasePtr<PyCodeObject>;

} // namespace NPython2
