#include <solomon/agent/lib/python2/cast_proto.h>
#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/gil.h>
#include <solomon/agent/lib/python2/initializer.h>
#include <solomon/agent/lib/python2/ptr.h>
#include <solomon/agent/lib/python2/ut/protos/cast_proto_test.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/tempfile.h>
#include <util/generic/ylimits.h>

using namespace NPython2;

class TCastProtoTest: public ::testing::Test {
private:
    TInitializer* Initializer_{TInitializer::Instance()};
};

TObjectPtr CallPyFunc(TStringBuf py, const char* fnName) {
    TGilGuard gil;

    TTempFileHandle tempFile;
    tempFile.Write(py.data(), py.size());
    tempFile.FlushData();

    TObjectPtr module = LoadPyModule("test_module", tempFile.Name());

    TObjectPtr func = PyObject_GetAttrString(module.Get(), fnName);
    Y_ENSURE(func, "cannot find function " << fnName);
    Y_ENSURE(PyCallable_Check(func.Get()), fnName << " is not callable object");

    TObjectPtr args = PyTuple_New(0);
    return PyObject_Call(func.Get(), args.Get(), nullptr);
}

TEST_F(TCastProtoTest, SinglePrimitives) {
    constexpr TStringBuf testPy =
            "def single_primitives():\n"
            "    return {\n"
            "        'Int32': 1,\n"
            "        'Int64': 2L,\n"
            "        'UInt32': 2,\n"
            "        'UInt64': 3L,\n"
            "        'Float': 3.14,\n"
            "        'Double': 6.28,\n"
            "        'Bool': True,\n"
            "        'Enum': 'TWO',\n"
            "        'String': 'some string',\n"
            "    }"sv;

    TObjectPtr pyObj = CallPyFunc(testPy, "single_primitives");
    TSinglePrimitives primitives;
    TProtoConverter converter(&primitives);
    converter.Convert(pyObj.Get());

    ASSERT_EQ(primitives.GetInt32(), 1);
    ASSERT_EQ(primitives.GetInt64(), 2);
    ASSERT_EQ(primitives.GetUInt32(), 2u);
    ASSERT_EQ(primitives.GetUInt64(), 3u);
    ASSERT_DOUBLE_EQ(primitives.GetFloat(), 3.14f);
    ASSERT_DOUBLE_EQ(primitives.GetDouble(), 6.28);
    ASSERT_TRUE(primitives.GetBool());
    ASSERT_EQ(primitives.GetEnum(), ETestEnum::TWO);
    ASSERT_EQ(primitives.GetString(), "some string");
}

TEST_F(TCastProtoTest, RepeatedPrimitives) {
    constexpr TStringBuf testPy =
            "def repeated_primitives():\n"
            "    return {\n"
            "        'Int32': [ 0x80000000, -1, 0, 1, 0x7fffffff ],\n"
            "        'Int64': [ 0x8000000000000000L, -1L, 0L, 1L, 0x7fffffffffffffffL ],\n"
            "        'UInt32': [ 0x80000000, -1, 0, 1, 0x7fffffff ],\n"
            "        'UInt64': [ 0x8000000000000000L, -1L, 0L, 1L, 0x7fffffffffffffffL ],\n"
            "        'Float': [ 1.17549435082228750797e-38, 0.0, 3.40282346638528859812e+38 ],\n"
            "        'Double': [ 2.22507385850720138309e-308, 0.0, 1.79769313486231570815e+308 ],\n"
            "        'Bool': [ True, False ],\n"
            "        'Enum': [ 'ZERO', 'ONE', 'TWO', 'THREE' ],\n"
            "        'String': [ 'str1', 'str2' ],\n"
            "    }"sv;

    TObjectPtr pyObj = CallPyFunc(testPy, "repeated_primitives");
    TRepeatedPrimitives primitives;
    TProtoConverter converter(&primitives);
    converter.Convert(pyObj.Get());

    {
        ASSERT_EQ(primitives.Int32Size(), 5u);
        ASSERT_EQ(primitives.GetInt32(0), Min<i32>());
        ASSERT_EQ(primitives.GetInt32(1), -1);
        ASSERT_EQ(primitives.GetInt32(2), 0);
        ASSERT_EQ(primitives.GetInt32(3), 1);
        ASSERT_EQ(primitives.GetInt32(4), Max<i32>());
    }
    {
        ASSERT_EQ(primitives.Int64Size(), 5u);
        ASSERT_EQ(primitives.GetInt64(0), Min<i64>());
        ASSERT_EQ(primitives.GetInt64(1), -1);
        ASSERT_EQ(primitives.GetInt64(2), 0);
        ASSERT_EQ(primitives.GetInt64(3), 1);
        ASSERT_EQ(primitives.GetInt64(4), Max<i64>());
    }
    {
        ASSERT_EQ(primitives.UInt32Size(), 5u);
        ASSERT_EQ(primitives.GetUInt32(0), ui32(0x80000000));
        ASSERT_EQ(primitives.GetUInt32(1), Max<ui32>());
        ASSERT_EQ(primitives.GetUInt32(2), 0u);
        ASSERT_EQ(primitives.GetUInt32(3), 1u);
        ASSERT_EQ(primitives.GetUInt32(4), ui32(0x7fffffff));
    }
    {
        ASSERT_EQ(primitives.UInt64Size(), 5u);
        ASSERT_EQ(primitives.GetUInt64(0), ui64(0x8000000000000000));
        ASSERT_EQ(primitives.GetUInt64(1), Max<ui64>());
        ASSERT_EQ(primitives.GetUInt64(2), 0u);
        ASSERT_EQ(primitives.GetUInt64(3), 1u);
        ASSERT_EQ(primitives.GetUInt64(4), ui64(0x7fffffffffffffff));
    }
    {
        ASSERT_EQ(primitives.FloatSize(), 3u);
        ASSERT_DOUBLE_EQ(primitives.GetFloat(0), Min<float>());
        ASSERT_DOUBLE_EQ(primitives.GetFloat(1), 0.0);
        ASSERT_DOUBLE_EQ(primitives.GetFloat(2), Max<float>());
    }
    {
        ASSERT_EQ(primitives.DoubleSize(), 3u);
        ASSERT_DOUBLE_EQ(primitives.GetDouble(0), Min<double>());
        ASSERT_DOUBLE_EQ(primitives.GetDouble(1), 0.0);
        ASSERT_DOUBLE_EQ(primitives.GetDouble(2), Max<double>());
    }
    {
        ASSERT_EQ(primitives.BoolSize(), 2u);
        ASSERT_TRUE(primitives.GetBool(0));
        ASSERT_FALSE(primitives.GetBool(1));
    }
    {
        ASSERT_EQ(primitives.EnumSize(), 4u);
        ASSERT_EQ(primitives.GetEnum(0), ETestEnum::ZERO);
        ASSERT_EQ(primitives.GetEnum(1), ETestEnum::ONE);
        ASSERT_EQ(primitives.GetEnum(2), ETestEnum::TWO);
        ASSERT_EQ(primitives.GetEnum(3), ETestEnum::THREE);
    }
    {
        ASSERT_EQ(primitives.StringSize(), 2u);
        ASSERT_EQ(primitives.GetString(0), "str1");
        ASSERT_EQ(primitives.GetString(1), "str2");
    }
}

TEST_F(TCastProtoTest, ComplexMessage) {
    constexpr TStringBuf testPy =
            "def permissions():\n"
            "    return {\n"
            "        'User': { 'Name': 'Jamel', 'Age': 97 },\n"
            "        'Permissions': [ 'CREATE', 'DELETE' ],\n"
            "        'Params': {'one': 'uno', 'two': 'dos', 'three': 'tres'},\n"
            "    }"sv;

    TObjectPtr pyObj = CallPyFunc(testPy, "permissions");
    TComplexMessage message;
    TProtoConverter converter(&message);
    converter.Convert(pyObj.Get());

    const auto& user = message.GetUser();
    ASSERT_EQ(user.GetName(), "Jamel");
    ASSERT_EQ(user.GetAge(), 97);

    const auto& permissions = message.GetPermissions();
    ASSERT_EQ(permissions.size(), 2);
    ASSERT_EQ(permissions.Get(0), EUserPermission::CREATE);
    ASSERT_EQ(permissions.Get(1), EUserPermission::DELETE);

    const auto& params = message.GetParams();
    ASSERT_EQ(params.size(), 3u);
    ASSERT_EQ(params.at("one"), "uno");
    ASSERT_EQ(params.at("two"), "dos");
    ASSERT_EQ(params.at("three"), "tres");
}
