#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/gil.h>
#include <solomon/agent/lib/python2/initializer.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/tempfile.h>

using namespace NPython2;

class TPython2MemoryTest: public ::testing::Test {
public:
    TPython2MemoryTest() {
        TString metricName = "python.AllocatedBytes";
        auto* registry = NMonitoring::TMetricRegistry::Instance();
        Raw_ = registry->IntGauge({ {"sensor", metricName}, {"domain", "Raw"} });
        Mem_ = registry->IntGauge({ {"sensor", metricName}, {"domain", "Mem"} });
        Obj_ = registry->IntGauge({ {"sensor", metricName}, {"domain", "Obj"} });
    }

    ui64 GetMemoryUsage() {
        return Raw_->Get() + Mem_->Get() + Obj_->Get();
    }

    ui64 CreateArray(TObjectPtr module, size_t elementsNum) {
        ui64 memoryBefore = GetMemoryUsage();

        TObjectPtr func = PyObject_GetAttrString(module.Get(), "get_array");
        Y_ENSURE(func);
        Y_ENSURE(PyCallable_Check(func.Get()));

        TObjectPtr args = Py_BuildValue("(I)", elementsNum);
        TObjectPtr result = PyObject_Call(func.Get(), args.Get(), nullptr);
        Y_ENSURE(result);

        ui64 memoryAfter = GetMemoryUsage();
        return memoryAfter - memoryBefore;
    }

    TObjectPtr TestCodeModule() {
        constexpr TStringBuf TestPy =
                "def get_number():\n"
                "    return 42\n"
                "\n"
                "def get_array(n):\n"
                "    return [12345] * n\n"sv;

        TTempFileHandle tempFile;
        tempFile.Write(TestPy.data(), TestPy.size());
        tempFile.FlushData();

        TObjectPtr module = LoadPyModule("test_init_module", tempFile.Name());
        Y_ENSURE(module);
        return module;
    }

private:
    TInitializer* Initializer_{TInitializer::Instance()};
    NMonitoring::TIntGauge* Raw_;
    NMonitoring::TIntGauge* Mem_;
    NMonitoring::TIntGauge* Obj_;
};

TEST_F(TPython2MemoryTest, IsMemoryTracked) {
    TGilGuard gil;

    TObjectPtr module = TestCodeModule();
    ui64 memInit = GetMemoryUsage();

    {
        TObjectPtr func = PyObject_GetAttrString(module.Get(), "get_number");
        ASSERT_TRUE(func);
        ASSERT_TRUE(PyCallable_Check(func.Get()));

        TObjectPtr result = PyObject_Call(func.Get(), PyTuple_New(0), nullptr);
        ASSERT_TRUE(result);
        ASSERT_TRUE(PyInt_Check(result.Get()));

        ui64 memAfterCreation = GetMemoryUsage();
        ASSERT_GT(memAfterCreation, memInit) << "Allocation of a small object";
    }

    {
        TGilUnguard ungil;
        ui64 memB = GetMemoryUsage();
        Sleep(TDuration::Seconds(3));

        ui64 memA = GetMemoryUsage();
        ASSERT_EQ(memA, memB);
    }

    {
        ui64 delta1 = CreateArray(module, 100);
        ui64 delta2 = CreateArray(module, 500);
        ui64 delta3 = CreateArray(module, 1000);

        ASSERT_LT(0u, delta1);
        ASSERT_LT(delta1, delta2);
        ASSERT_LT(delta2, delta3);
    }
}
