#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/error.h>
#include <solomon/agent/lib/python2/gil.h>
#include <solomon/agent/lib/python2/initializer.h>
#include <solomon/agent/lib/python2/logger.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/logger_test.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/str.h>
#include <util/string/split.h>
#include <util/system/tempfile.h>

using namespace NPython2;

TObjectPtr CallPyFunc(const char* fnName, TObjectPtr args) {
    constexpr TStringBuf moduleText =
            "def test(logger):\n"
            "    logger.trace('trace message')\n"
            "    logger.debug('debug message')\n"
            "    logger.info('info message')\n"
            "    logger.warn('warning message')\n"
            "    logger.error('error message')\n"
            "    logger.fatal('fatal message')\n"
    ;

    TTempFileHandle tempFile;
    tempFile.Write(moduleText.data(), moduleText.size());
    tempFile.FlushData();

    TGilGuard gil;

    TObjectPtr module = LoadPyModule("test_module", tempFile.Name());
    TObjectPtr func = PyObject_GetAttrString(module.Get(), fnName);
    Y_ENSURE(func);
    Y_ENSURE(PyCallable_Check(func.Get()));

    TObjectPtr res = PyObject_Call(func.Get(), args.Get(), nullptr);
    Y_ENSURE(res && res.Get() == Py_None, "python error: " << LastErrorAsString());

    return res;
}

TEST(TPyLoggerTest, Levels) {
    using NSolomon::NAgent::ELogLevel;

    TInitializer::Instance();

    TStringStream out;
    NSolomon::NAgent::InitLogger(&out); // default level is INFO

    TObjectPtr logger = GetLogger();
    CallPyFunc("test", PyTuple_Pack(1, logger.Get()));

    TString infoStr, warnStr, errorStr, fatalStr, _;
    Split(out.Str(), '\n', infoStr, warnStr, errorStr, fatalStr, _);

    {
        TLogRow logRow = ParseLogRow(infoStr);
        ASSERT_EQ(logRow.Level, ELogLevel::INFO);
        ASSERT_EQ(logRow.FileName, "python2");
        ASSERT_TRUE(logRow.LineNumber == 0);
        ASSERT_EQ(logRow.Message, "info message");
    }
    {
        TLogRow logRow = ParseLogRow(warnStr);
        ASSERT_EQ(logRow.Level, ELogLevel::WARN);
        ASSERT_EQ(logRow.FileName, "python2");
        ASSERT_TRUE(logRow.LineNumber == 0);
        ASSERT_EQ(logRow.Message, "warning message");
    }
    {
        TLogRow logRow = ParseLogRow(errorStr);
        ASSERT_EQ(logRow.Level, ELogLevel::ERROR);
        ASSERT_EQ(logRow.FileName, "python2");
        ASSERT_TRUE(logRow.LineNumber == 0);
        ASSERT_EQ(logRow.Message, "error message");
    }
    {
        TLogRow logRow = ParseLogRow(fatalStr);
        ASSERT_EQ(logRow.Level, ELogLevel::FATAL);
        ASSERT_EQ(logRow.FileName, "python2");
        ASSERT_TRUE(logRow.LineNumber == 0);
        ASSERT_EQ(logRow.Message, "fatal message");
    }
}
