#pragma once

#include <solomon/agent/lib/auth/auth.h>
#include <solomon/agent/lib/thread/pool_provider.h>
#include <solomon/agent/misc/timer_thread.h>
#include <solomon/agent/protos/agent_config.pb.h>

#include <solomon/libs/cpp/backoff/backoff.h>
#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/cloud/iam/client.h>
#include <solomon/libs/cpp/cloud/iam/iam.h>
#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/push_client/auth.h>
#include <library/cpp/threading/future/future.h>

#include <util/system/spinlock.h>

namespace NSolomon::NAgent {

class TRegistrationTask: public TAtomicRefCount<TRegistrationTask> {
public:
    TRegistrationTask(
            const TAgentConfig& config,
            NMonitoring::TMetricRegistry& registry,
            TTimerThread& timer,
            TSimpleSharedPtr<IThreadPool> pool);
    ~TRegistrationTask();

    void Start();
    void Stop();

    bool IsRegisteredIn(const TString& clusterName) const;

private:
    struct TClusterRegistrationState: public TAtomicRefCount<TClusterRegistrationState> {
        TClusterRegistrationState(const TString& endpoint, NSolomon::IAuthProviderPtr authProvider)
            : Endpoint(endpoint)
            , IsRegistered{true}
            , AuthProvider{std::move(authProvider)}
        {}

        TString Endpoint;
        TDuration PreviousDelay;
        std::atomic_bool IsRegistered{true}; // assume that we're running in a new mode right at startup
        NSolomon::IAuthProviderPtr AuthProvider;
        NThreading::TFuture<void> CurrentRequest;
    };

    using TClusterRegistrationStatePtr = TIntrusivePtr<TClusterRegistrationState>;

    std::function<void()> CreateTimerFuncClosure(TString clusterName, TClusterRegistrationStatePtr state);

    NThreading::TFuture<ui64> SendRequest(
            TString clusterName,
            TClusterRegistrationStatePtr state);

    // "bind before connect" trick: we will specify a source IP manually to save a real client IP.
    // For more info see SOLOMON-4976
    void InitHttpClient(NMonitoring::TMetricRegistry& registry, bool bindBeforeConnect = false);

private:
    IHttpClientPtr HttpClient_;
    TRequestOpts ReqOpts_;
    TString RegistrationData_;

    THashMap<TString, TClusterRegistrationStatePtr> ClusterNameToState_;

    TTimerThread& TimerThread_;
    TSimpleSharedPtr<IThreadPool> ThreadPool_;
    THalfJitter Jitter_;
    TExpBackoff<THalfJitter> Backoff_;

    THashMap<EClusterType, TString> ClustersHosts_;
};

using TRegistrationTaskPtr = TIntrusivePtr<TRegistrationTask>;

} // namespace NSolomon::NAgent
