#include "modules_page.h"

#include <solomon/agent/lib/selfmon/status/puller_status.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/json/writer/json.h>

#include <util/generic/maybe.h>
#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {

TModuleStatePage::TModuleStatePage(const TPullerStatus& status)
    : TPreMonPage{"modules", "Modules", false}
    , Status_{status}
{
}

void TModuleStatePage::OutputText(IOutputStream& os, NMonitoring::IMonHttpRequest&) {
    static TString SUCCESS = "SUCCESS";

    HTML(os) {
        TABLE_CLASS("table table-condensed") {
            CAPTION() {
                os << "Agent modules' status";
            }
            TABLEHEAD() {
                TABLER() {
                    TABLEH() {
                        os << "Name";
                    }

                    TABLEH() {
                        os << "Pull interval";
                    }

                    TABLEH() {
                        os << "Completed at";
                    }

                    TABLEH() {
                        os << "Return value";
                    }

                    TABLEH() {
                        os << "Execution time";
                    }

                    TABLEH() {
                        os << "State";
                    }

                    TABLEH() {
                        os << "Scheduler state";
                    }
                }
            }

            TABLEBODY() {
                auto writeRow = [&] (const auto& module) {
                    TABLER() {
                        TABLED() {
                            os << module.Name;
                        }
                        TABLED() {
                            os << module.Interval.Seconds();
                        }
                        TABLED() {
                            os << module.Timestamp.ToStringLocal();
                        }
                        TABLED() {
                            os << module.ReturnValue;
                        }
                        TABLED() {
                            os << module.ExecTime.MilliSeconds();
                        }
                        TABLED() {
                            os << (module.ErrorMessage.Empty()
                                ? SUCCESS
                                : TStringBuilder() << "FAILURE: " << *module.ErrorMessage);
                        }
                        TABLED() {
                            os << module.SchedulerState;
                        }
                    }
                };

                Status_.ForEachModule(writeRow);
            }
        }
    }
}

void TModuleStatePage::OutputJson(NMonitoring::IMonHttpRequest& request) {
    request.Output().Write(HTTPOKJSON);

    NJsonWriter::TBuf json{NJsonWriter::HEM_RELAXED, &request.Output()};
    json.BeginList();

    auto writeOne = [&] (const auto& module) {
        json.BeginObject();
        json.WriteKey("Name");
        json.WriteString(module.Name);
        json.WriteKey("LastRunTimestamp");
        json.WriteInt(module.Timestamp.Seconds());
        json.WriteKey("LastReturnValue");
        json.WriteInt(module.ReturnValue);
        json.WriteKey("IntervalSeconds");
        json.WriteInt(module.Interval.Seconds());
        json.WriteKey("Success");
        json.WriteBool(module.ErrorMessage.Empty());
        json.WriteKey("ExecTimeMilliseconds");
        json.WriteInt(module.ExecTime.MilliSeconds());
        json.WriteKey("SchedulerState");
        json.WriteString(::ToString(module.SchedulerState));
        json.WriteKey("ErrorMessage");
        json.WriteString(module.ErrorMessage ? *module.ErrorMessage : TString{});
        json.EndObject();

    };

    Status_.ForEachModule(writeOne);

    json.EndList();
}

void TModuleStatePage::Output(NMonitoring::IMonHttpRequest& request) {
    TVector<TStringBuf> parts;
    StringSplitter(request.GetPathInfo()).Split('/').SkipEmpty().Collect(&parts);

    if (parts.empty()) {
        TPreMonPage::Output(request);
    } else if (parts.back() == "json") {
        OutputJson(request);
    } else {
        NotFound(request);
    }
}

} // namespace NAgent
} // namespace NSolomon
