#include "status.h"

#include <solomon/agent/lib/storage/memory_usage.h>

using NMonitoring::TMetricRegistry;


namespace NSolomon::NAgent {
namespace {

struct TMemoryUsageInfoListener: public IMemoryUsageInfoListener {
public:
    TMemoryUsageInfoListener(TMetricRegistry& metricRegistry)
        : Registry_{metricRegistry}
    {
        CurrentSizeBytes_ = Registry_.IntGauge({ {"sensor", "storage.sizeBytes"}, {"storageShard", "total"} });
        LimitTotalBytes_ = Registry_.IntGauge({ {"sensor", "storage.limitBytes"}, {"storageShard", "total"} });
        BytesWritten_ = Registry_.Counter({ {"sensor", "storage.bytesWritten"}, {"storageShard", "total"} });
        BytesEvicted_ = Registry_.Counter({ {"sensor", "storage.bytesEvicted"}, {"storageShard", "total"} });
        BytesRead_ = Registry_.Counter({ {"sensor", "storage.bytesRead"}, {"storageShard", "total"} });
    }

    void OnSizeChanged(TBytes memorySize) noexcept override {
        CurrentSizeBytes_->Set(memorySize);
    }

    void OnBytesWritten(TBytes memorySize) noexcept override {
        BytesWritten_->Add(memorySize);
    }

    void OnBytesEvicted(TBytes memorySize) noexcept override {
        BytesEvicted_->Add(memorySize);
    }

    void OnBytesRead(TBytes memorySize) noexcept override {
        BytesRead_->Add(memorySize);
    }

    void OnLimitSet(TBytes memorySize) noexcept override {
        LimitTotalBytes_->Set(static_cast<double>(memorySize));
    }

private:
    TMetricRegistry& Registry_;

    NMonitoring::TIntGauge* CurrentSizeBytes_{nullptr};
    NMonitoring::TIntGauge* LimitTotalBytes_{nullptr};
    NMonitoring::TCounter* BytesWritten_{nullptr};
    NMonitoring::TCounter* BytesEvicted_{nullptr};
    NMonitoring::TCounter* BytesRead_{nullptr};
};

} // namespace

IMemoryUsageInfoListenerPtr CreateMemoryUsageInfoListener(TMetricRegistry& metricRegistry) {
    return ::MakeHolder<TMemoryUsageInfoListener>(metricRegistry);
}

} // namespace NSolomon::NAgent
