#include "status.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <solomon/agent/misc/background_threads.h>
#include <solomon/agent/misc/logger.h>

#include <solomon/libs/cpp/process_stats/process_stats.h>

#include <util/system/getpid.h>
#include <util/stream/file.h>
#include <util/folder/path.h>
#include <util/string/split.h>

namespace NSolomon {
namespace NAgent {
namespace {

struct TProcessStatsEx: TProcessStats {
    ui64 Uptime {0};
};

using namespace NMonitoring;

class TProcessStatus: public TBackgroundThreads {
public:
    explicit TProcessStatus(TMetricRegistry& registry)
        : Registry_{registry}
    {
        static constexpr auto TIMEOUT = TDuration::Seconds(5);

        RunInBackground(
            TStringBuf("ProcessStatus"),
            [&] {
                const auto stats = ParseProcessStats();
                WriteStats(stats);
                LastStats_ = stats;
                return TIMEOUT;
            },
            TDuration::Zero()
        );
    }

    ~TProcessStatus() {
        Stop();
    }

private:
    TProcessStatsEx ParseProcessStats() const noexcept {
        auto result = StatProvider_->GetSelfStats();
        if (!result.Success()) {
            SA_LOG(WARN) << "Error during process stats collection: " << result.Error().Message();
            return {};
        }

        TProcessStatsEx stats{result.Extract()};
        stats.Uptime = (TInstant::Now() - Start_).MilliSeconds();

        return stats;
    }

    void WriteStats(const TProcessStatsEx& stats) {
        Registry_.IntGauge({{"sensor", "process.memRssBytes"}})->Set(stats.MemRss);
        Registry_.IntGauge({{"sensor", "process.memLibBytes"}})->Set(stats.MemLib);
        Registry_.IntGauge({{"sensor", "process.memSwapBytes"}})->Set(stats.MemSwap);
        Registry_.IntGauge({{"sensor", "process.threadCount"}})->Set(stats.ThreadCount);
        Registry_.IntGauge({{"sensor", "process.uptimeMillis"}})->Set(stats.Uptime);
        Registry_.Rate({{"sensor", "process.majorPageFaults"}})->Add(stats.MajorPageFaults - LastStats_.MajorPageFaults);
        Registry_.Rate({{"sensor", "process.cpuUserMillis"}})->Add(stats.CpuUser - LastStats_.CpuUser);
        Registry_.Rate({{"sensor", "process.cpuSystemMillis"}})->Add(stats.CpuSystem - LastStats_.CpuSystem);
    }

private:
    TInstant Start_{TInstant::Now()};
    const TFsPath ProcessStatsBase_{"/proc/self"};
    IProcessStatProviderPtr StatProvider_{CreateProcessStatProvider()};
    TMetricRegistry& Registry_;
    TProcessStatsEx LastStats_;
};

} // namespace

THolder<TBackgroundThreads> CreateProcessMonitoringThread(NMonitoring::TMetricRegistry& registry) {
    return ::MakeHolder<TProcessStatus>(registry);
}

} // namespace NAgent
} // namespace NSolomon
