#include "puller_status.h"

#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/labels.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/cast.h>

namespace NSolomon {
namespace NAgent {

using NMonitoring::TMetricRegistry;

TPullerStatus::TPullerStatus(TMetricRegistry& registry)
    : Registry_{registry}
{
}

void TPullerStatus::UpdateModuleData(TModuleStatus data) noexcept {
    try {
        Registry_.Gauge({{"module", data.Name}, {"sensor", "execTimeMs"}})->Set(data.ExecTime.MilliSeconds());
    } catch (...) {
        SA_LOG(ERROR) << ::CurrentExceptionMessage();
        return;
    }

    auto g = Guard(Lock_);
    ModulesStatus_[data.Name] = data;
}

void TPullerStatus::UpdateModuleDataOnSkip(TStringBuf name, TDuration waitTime) noexcept {
    if (waitTime) {
        Registry_.Gauge({{"module", name}, {"sensor", "waitTimeMs"}})->Set(waitTime.MilliSeconds());
    }

    Registry_.Counter({{"module", name}, {"sensor", "skipped"}})->Inc();
}

void TPullerStatus::UpdateModuleState(TStringBuf name, TTaskState::EState state) noexcept {
    auto g = Guard(Lock_);
    ModulesStatus_[name].SchedulerState = state;
}

void TPullerStatus::RemoveModuleData(TStringBuf name) {
    with_lock (Lock_) {
        ModulesStatus_.erase(name);
    }

    Registry_.RemoveMetric(TAgentLabels{{"module", name}, {"sensor", "execTimeMs"}});
}

void TPullerStatus::ForEachModule(const std::function<void(const TModuleStatus&)>& fn) const {
    auto g = Guard(Lock_);
    for (const auto& pair: ModulesStatus_) {
        const auto& module = pair.second;
        fn(module);
    }
}

} // NAgent
} // NSolomon
