#include "status.h"

#include <solomon/agent/lib/pusher/pusher.h>

#include <solomon/libs/cpp/sync/rw_lock.h>

namespace NSolomon::NAgent {
namespace {

using namespace NMonitoring;

struct TMappingKey {
    TShardKey ShardKey;
    TString HostUrl;

    operator size_t() const {
        return MultiHash(TShardKeyHasher{}(ShardKey), HostUrl);
    }
};

class THandleMetrics: public TAtomicRefCount<THandleMetrics> {
public:
    THandleMetrics(TMetricRegistry& registry, const std::function<NMonitoring::TLabels(TStringBuf)>& makeLabels)
        : RequestOk{registry.Rate(makeLabels(TStringBuf("requestOk")))}
        , RequestError{registry.Rate(makeLabels(TStringBuf("requestError")))}
        , RespTimes{
            registry.HistogramCounter(makeLabels(TStringBuf("responseTimeMillis")),
            ExponentialHistogram(13, 2, 16))
        }
        , OutstandingRequests{registry.IntGauge(makeLabels(TStringBuf("outstanding")))}
    {
    }

    void WriteRespTime(TDuration respTime) noexcept {
        RespTimes->Record(respTime.MilliSeconds());
    }

    void UpdateSuccessAndErrorCounters(bool hasFailed) {
        if (!hasFailed) {
            RequestOk->Inc();
        } else {
            RequestError->Inc();
        }
    }

    void RequestStarted() {
        OutstandingRequests->Inc();
    }

    void RequestFinished() {
        OutstandingRequests->Dec();
    }

private:
    TRate* RequestOk;
    TRate* RequestError;
    THistogram* RespTimes;
    TIntGauge* OutstandingRequests;
};

class DataPusherStatus: public IDataPusherStatusListener, TMoveOnly {
public:
    DataPusherStatus(TMetricRegistry& metricRegistry, NMonitoring::TLabels additionalLabels)
        : Registry_(metricRegistry)
        , AdditionalLabels_(std::move(additionalLabels))
    {
    }

    void OnRequestCompleted(
            const TShardKey& shardKey,
            TString hostUrl,
            bool hasFailed,
            TDuration respTime) noexcept override
    {
        auto writeStats = [&](TIntrusivePtr<THandleMetrics> metrics) {
            metrics->WriteRespTime(respTime);
            metrics->UpdateSuccessAndErrorCounters(hasFailed);
            metrics->RequestFinished();
        };

        writeStats(GetOrCreateHandleMetrics({shardKey, hostUrl}));
        writeStats(GetOrCreateHandleMetrics({SummaryMetricsKey_, hostUrl}));
    }

    void OnRequestStarted(const TShardKey& shardKey, TString hostUrl) noexcept override {
        auto writeStats = [](TIntrusivePtr<THandleMetrics> metrics) {
            metrics->RequestStarted();
        };

        writeStats(GetOrCreateHandleMetrics({shardKey, hostUrl}));
        writeStats(GetOrCreateHandleMetrics({SummaryMetricsKey_, hostUrl}));
    }

private:
    TIntrusivePtr<THandleMetrics> CreateHandleMetrics(const TShardKey& shardKey, TStringBuf hostUrl) {
        auto makeLabels = [=](TStringBuf metricName) -> NMonitoring::TLabels {
            NMonitoring::TLabels labels{AdditionalLabels_};
            labels.Add(TStringBuf("projectId"), shardKey.Project);
            labels.Add(TStringBuf("clusterId"), shardKey.Cluster);
            labels.Add(TStringBuf("serviceId"), shardKey.Service);
            labels.Add(TStringBuf("hostUrl"), hostUrl);
            labels.Add(TStringBuf("sensor"), TStringBuilder{} << "dataPusher." << metricName);

            return labels;
        };

        return MakeIntrusive<THandleMetrics>(Registry_, makeLabels);
    }

    TIntrusivePtr<THandleMetrics> GetOrCreateHandleMetrics(const TMappingKey& key) {
        {
            auto mapping = Mapping_.Read();
            if (auto it = mapping->find(key); it != mapping->end()) {
                return it->second;
            }
        }

        auto handleMetrics = CreateHandleMetrics(key.ShardKey, key.HostUrl);

        auto mapping = Mapping_.Write();
        auto [it, _] = mapping->emplace(key, handleMetrics);
        return it->second;
    }

private:
    TMetricRegistry& Registry_;
    const NMonitoring::TLabels AdditionalLabels_;
    const TShardKey SummaryMetricsKey_{"total", "total", "total"};
    NSync::TLightRwLock<THashMap<TMappingKey, TIntrusivePtr<THandleMetrics>>> Mapping_;
};

} // namespace

IDataPusherStatusListenerPtr CreateDataPusherStatusListener(
    TMetricRegistry& metricRegistry,
    NMonitoring::TLabels additionalLabels)
{
    return ::MakeAtomicShared<DataPusherStatus>(metricRegistry, std::move(additionalLabels));
}

} // namespace NSolomon::NAgent
