#include "status.h"

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/histogram_collector.h>

#include <util/string/builder.h>
#include <util/string/subst.h>

namespace NSolomon {
namespace NAgent {

using namespace NMonitoring;

class TServerStatus: public IServerStatusListener {
private:
    struct TTimeBucket {
        TRate* Metric;
        ui64 ThresholdMs;
    };

    struct THandleMetrics: public TAtomicRefCount<THandleMetrics> {
        THandleMetrics(TMetricRegistry& registry, const std::function<NMonitoring::TLabels(TStringBuf)>& makeLabels)
            : RequestRate{registry.Rate(makeLabels(TStringBuf("requestRate")))}

            , Ok{registry.Rate(makeLabels(TStringBuf("2xx")))}
            , Error{registry.Rate(makeLabels(TStringBuf("error")))}
            , ClientError{registry.Rate(makeLabels(TStringBuf("4xx")))}
            , ServerError{registry.Rate(makeLabels(TStringBuf("5xx")))}
            , StatusMetricFactory{[&registry, makeLabels] (auto&& name) {
                return registry.Rate(makeLabels(name));
            }}
            , ResponseBytes{registry.Rate(makeLabels(TStringBuf("responseBytes")))}
            , RespTimes{
                registry.HistogramRate(makeLabels(TStringBuf("responseTimeMillis")),
                ExponentialHistogram(20, 2, 2))}
        {
        }

        void WriteRespTime(TDuration time) noexcept {
            RespTimes->Record(time.MilliSeconds());
        }

        void WriteCode(HttpCodes code) noexcept {
            const auto codeBucket = code / 100;

            if (codeBucket == 2) {
                Ok->Inc();
                return;
            }

            Error->Inc();
            if (codeBucket == 4) {
                ClientError->Inc();
            } else if (codeBucket == 5) {
                ServerError->Inc();
            } else {
                StatusMetricFactory(::ToString<std::underlying_type<HttpCodes>::type>(code))->Inc();
            }
        }

        TRate* RequestRate;

        TRate* Ok;
        TRate* Error;
        TRate* ClientError;
        TRate* ServerError;
        std::function<TRate*(const TString&)> StatusMetricFactory;

        TRate* ResponseBytes;
        THistogram* RespTimes;
    };

    using TMapping = THashMap<TString, TIntrusivePtr<THandleMetrics>>;

public:
    explicit TServerStatus(TMetricRegistry& registry, NMonitoring::TLabels additionalLabels)
        : Registry_{registry}
        , AdditionalLabels_{std::move(additionalLabels)}
    {
    }

    void OnRequestCompleted(TStringBuf path, HttpCodes code, TDuration time, ui64 responseSize) noexcept override {
        TIntrusivePtr<THandleMetrics> metrics;
        {
            auto mapping = Mapping_.Read();
            if (auto it = mapping->find(path); it != mapping->end()) {
                metrics = it->second;
            }
        }

        if (!metrics) {
            auto handlerMetrics = CreateHandleMetrics(path);

            auto mapping = Mapping_.Write();
            auto [it, _] = mapping->emplace(path, handlerMetrics);
            metrics = it->second;
        }

        metrics->WriteRespTime(time);
        metrics->WriteCode(code);
        metrics->RequestRate->Inc();
        metrics->ResponseBytes->Add(responseSize);
    }

    void OnRequest(TStringBuf /*path*/, ui64 /*payloadSize*/) noexcept override {
    }

private:
    TIntrusivePtr<THandleMetrics> CreateHandleMetrics(TStringBuf path) {
        TString name{path};

        // ensure all values start and end with a point
        if (!name.StartsWith('/')) {
            name.prepend('.');
        }

        if (!name.EndsWith('/')) {
            name.append('.');
        }

        SubstGlobal(name, '/', '.');

        auto makeLabels = [this, &name] (TStringBuf metric) {
            NMonitoring::TLabels labels{AdditionalLabels_};
            labels.Add("sensor", TStringBuilder() << "http" << name << metric);
            return labels;
        };

        return MakeIntrusive<THandleMetrics>(Registry_, makeLabels);
    }

private:
    TMetricRegistry& Registry_;
    const NMonitoring::TLabels AdditionalLabels_;
    NSync::TLightRwLock<TMapping> Mapping_;
};

IServerStatusListenerPtr CreateServerStatusListener(TMetricRegistry& registry, NMonitoring::TLabels additionalLabels) {
    return ::MakeHolder<TServerStatus>(registry, std::move(additionalLabels));
}

} // NAgent
} // NSolomon
