#include "status.h"

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/strbuf.h>
#include <util/generic/ptr.h>

namespace NSolomon {
namespace NAgent {

using NMonitoring::TMetricRegistry;

struct TStorageShardStats: public IStorageUpdateListener {
public:
    TStorageShardStats(const TStorageShardId& shard, TMetricRegistry& registry)
        : Metrics_{registry}
        , ShardId_{shard.Project + "/" + shard.Service}
    {
        Chunks_ = Metrics_.Counter({{"sensor", "storage.chunkCount"}, {"storageShard", ShardId_}});
        ChunksEvictedOnOverflow_ = Metrics_.Counter({{"sensor", "storage.chunkEvictedOnOverflow"}, {"storageShard", ShardId_}});
        ChunksRemovedAsRead_ = Metrics_.Counter({{"sensor", "storage.chunksRemovedAsRead"}, {"storageShard", ShardId_}});
        Points_ = Metrics_.Counter({{"sensor", "storage.pointCount"}, {"storageShard", ShardId_}});

        CurrentSizeBytes_ = Metrics_.IntGauge({{"sensor", "storage.sizeBytes"}, {"storageShard", ShardId_}});
        PerShardLimitBytes_ = Metrics_.IntGauge({{"sensor", "storage.limitBytes"}, {"storageShard", ShardId_}});
        BytesWritten_ = Metrics_.Counter({{"sensor", "storage.bytesWritten"}, {"storageShard", ShardId_}});
        BytesEvicted_ = Metrics_.Counter({{"sensor", "storage.bytesEvicted"}, {"storageShard", ShardId_}});
        BytesRead_ = Metrics_.Counter({{"sensor", "storage.bytesRead"}, {"storageShard", ShardId_}});
    }

    void OnFetcherUpdated(const TString& fetcherId, ui64 chunkOffset) noexcept override {
        // race with UpdateLag is possible here, but we don't need complete consistency

        NMonitoring::TIntGauge* lagGauge = nullptr;
        {
            auto fetcherLag = FetcherLag_.Read();
            if (auto it = fetcherLag->find(fetcherId); it != fetcherLag->end()) {
                lagGauge = it->second;
            }
        }

        if (!lagGauge) {
            // metric registry already guaranties uniqueness
            lagGauge = Metrics_.IntGauge({{"fetcherId", fetcherId}, {"sensor", "storage.fetcher.lag"}});

            auto fetcherLag = FetcherLag_.Write();
            fetcherLag->emplace(fetcherId, lagGauge);
        }

        if (lagGauge->Get() < static_cast<i64>(chunkOffset)) {
            lagGauge->Set(0);
        } else {
            lagGauge->Add(-static_cast<i64>(chunkOffset));
        }
    }

    void OnPointsAdded(ui64 count) noexcept override {
        Points_->Add(count);
    }

    void OnPointsRemoved(ui64 count) noexcept override {
        Points_->Add(-count);
    }

    void OnSizeChanged(TBytes size) noexcept override {
        CurrentSizeBytes_->Set(size);
    }

    void OnBytesWritten(TBytes size) noexcept override {
        BytesWritten_->Add(size);
    }

    void OnBytesEvicted(TBytes size) noexcept override {
        BytesEvicted_->Add(size);
    }

    void OnBytesRead(TBytes size) noexcept override {
        BytesRead_->Add(size);
    }

    void OnAdded(ui64 chunkCount) noexcept override {
        UpdateLag(chunkCount);
        Chunks_->Add(chunkCount);
    }

    void OnOverflow(ui64 chunkCount) noexcept override {
        ChunksEvictedOnOverflow_->Add(chunkCount);
        Chunks_->Add(-chunkCount);
    }

    void OnRemovedReadChunks(ui64 chunkCount) noexcept override {
        ChunksRemovedAsRead_->Add(chunkCount);
        Chunks_->Add(-chunkCount);
    }

    void OnLimitSet(TBytes memorySize) noexcept override {
        PerShardLimitBytes_->Set(memorySize);
    }

private:
    void UpdateLag(i64 value) {
        auto fetcherLag = FetcherLag_.Read();
        for (auto& [_, lag]: *fetcherLag) {
            lag->Add(value);
        }
    }

private:
    NMonitoring::TMetricRegistry& Metrics_;
    NSync::TLightRwLock<THashMap<TString, NMonitoring::TIntGauge*>> FetcherLag_;

    TString ShardId_;
    std::atomic<ui64> EndSeqNo_{0};
    std::atomic<ui64> StartSeqNo_{0};

    NMonitoring::TCounter* Chunks_ {nullptr};
    NMonitoring::TCounter* Points_ {nullptr};
    NMonitoring::TCounter* ChunksEvictedOnOverflow_ {nullptr};
    NMonitoring::TCounter* ChunksRemovedAsRead_ {nullptr};
    NMonitoring::TIntGauge* CurrentSizeBytes_ {nullptr};
    NMonitoring::TIntGauge* PerShardLimitBytes_ {nullptr};
    NMonitoring::TCounter* BytesWritten_{nullptr};
    NMonitoring::TCounter* BytesEvicted_{nullptr};
    NMonitoring::TCounter* BytesRead_{nullptr};
};

IStorageUpdateListenerPtr CreateStorageShardUpdateListener(const TStorageShardId& shardId, TMetricRegistry& registry) {
    return ::MakeHolder<TStorageShardStats>(shardId, registry);
}

} // namespace NAgent
} // namespace NSolomon
