#include "status.h"

#include <solomon/libs/cpp/ewma/ewma.h>

#include <cmath>

namespace NSolomon::NAgent {
namespace {

using namespace NMonitoring;

class TThreadPoolStatus: public IThreadPoolStatusListener, TMoveOnly {
public:
    TThreadPoolStatus(TString poolName, TMetricRegistry& metricRegistry, NMonitoring::TLabels additionalLabels)
        : PoolName_{std::move(poolName)}
        , AdditionalLabels_(std::move(additionalLabels))
    {
        Metrics_.ActiveThreads = metricRegistry.IntGauge(MakeLabels(TStringBuf("activeThreads")));
        Metrics_.QueueSize = metricRegistry.IntGauge(MakeLabels(TStringBuf("queueSize")));
        Metrics_.WaitTime = metricRegistry.Gauge(MakeLabels(TStringBuf("waitTimeUs")));

        Ewma_ = NMonitoring::TEwmaMeter(MakeHolder<TEWMAMean>(Metrics_.WaitTime, EWMA_ALPHA_1MIN));
    }

private:
    void OnStart(size_t threadCount, size_t queueSizeLimit) noexcept override {
        SA_LOG(DEBUG) << "Thread pool " << PoolName_ << " has been started:"
                      << " threadCount=" << threadCount
                      << "; queueSizeLimit=" << queueSizeLimit
                      ;
    }

    void OnTaskScheduled() noexcept override {
        Metrics_.QueueSize->Inc();
    }

    void OnTaskRejected() noexcept override {
        SA_LOG(ERROR) << "Failed to push a task to the threadpool \"" << PoolName_ << "\"";
    }

    void OnTaskBeforeExecution(TDuration waitTime) noexcept override {
        // Note: without additional locking metrics will be updated independently
        // and form a state different from a real one
        Metrics_.ActiveThreads->Inc();
        Ewma_.Mark(waitTime.MicroSeconds());
    }

    void OnTaskCompleted(TDuration) noexcept override {
        Metrics_.ActiveThreads->Dec();
        Metrics_.QueueSize->Dec();
    }

    void OnStop() noexcept override {
        SA_LOG(DEBUG) << "Thread pool " << PoolName_ << " has been stopped";
    }

private:
    NMonitoring::TLabels MakeLabels(TStringBuf metricName) const {
        TString label = TStringBuilder() << "threadPool." << metricName;
        NMonitoring::TLabels labels{AdditionalLabels_};
        labels.Add("sensor", label);

        return labels;
    }

private:
    struct {
        TIntGauge* ActiveThreads = nullptr;
        TIntGauge* QueueSize = nullptr;
        TGauge* WaitTime = nullptr;
    } Metrics_;

    TString PoolName_;
    NMonitoring::TLabels AdditionalLabels_;

    NMonitoring::TEwmaMeter Ewma_{};
};

} // namespace

IThreadPoolStatusListenerPtr CreateThreadPoolStatusListener(
    TString poolName,
    TMetricRegistry& metricRegistry,
    NMonitoring::TLabels additionalLabels
) {
    return std::make_shared<TThreadPoolStatus>(std::move(poolName), metricRegistry, std::move(additionalLabels));
}

} // namespace NSolomon::NAgent
