#include "mem_storage.h"

#include <solomon/agent/misc/countdown_event.h>
#include <solomon/agent/misc/mutable_histogram.h>
#include <solomon/agent/misc/timer_dispatcher.h>

#include <library/cpp/messagebus/actor/actor.h>

#include <util/generic/hash_set.h>
#include <util/generic/ptr.h>
#include <util/generic/scope.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/thread/lfqueue.h>

#include <array>
#include <iterator>
#include <limits>

using NMonitoring::EMetricType;
using NMonitoring::IHistogramSnapshot;
using NMonitoring::ISummaryDoubleSnapshot;
using NMonitoring::TBucketBound;
using NMonitoring::TBucketBounds;
using NMonitoring::TBucketValue;
using NMonitoring::TBucketValues;
using NMonitoring::TMetricTimeSeries;
using TPoint = NMonitoring::TMetricTimeSeries::TPoint;

namespace NSolomon::NAgent {
namespace {

class TMutableSummaryDouble: public ISummaryDoubleSnapshot {
public:
    double GetSum() const noexcept override {
        return Sum_;
    }

    double GetMin() const noexcept override {
        return Min_;
    }

    double GetMax() const noexcept override {
        return Max_;
    }

    double GetLast() const noexcept override {
        return Last_;
    }

    ui64 GetCount() const noexcept override {
        return Count_;
    }

    ui64 MemorySizeBytes() const noexcept override {
        return sizeof(*this);
    }

    void Merge(ISummaryDoubleSnapshot* other) {
        Sum_ += other->GetSum();
        Min_ = std::min(Min_, other->GetMin());
        Max_ = std::max(Max_, other->GetMax());
        Last_ = other->GetLast();
        Count_ += other->GetCount();
    }

    void Reset() noexcept {
        *this = TMutableSummaryDouble();
    }

private:
    double Sum_ = 0;
    double Min_ = std::numeric_limits<double>::max();
    double Max_ = std::numeric_limits<double>::lowest();
    double Last_ = 0;
    ui64 Count_ = 0;
};

template <typename T>
struct TAggregate {
    ui64 Count = 0;
    T Sum = 0;
    // TODO:
    /* T Avg; */
    /* T Min; */
    /* T Max; */
};

template<>
struct TAggregate<IHistogramSnapshot*> {
    ui64 Count = 0;
    TMutableHistogramSnapshot Histogram;
};

template <>
struct TAggregate<ISummaryDoubleSnapshot*> {
    ui64 Count = 0;
    TMutableSummaryDouble Summary;
};

using TLabelsPtr = TAtomicSharedPtr<NWithMemoryInfo::TLabels>;

struct TLabelsPtrHasher {
    ui64 operator()(const TLabelsPtr& labels) {
        return labels->Hash();
    }
};

struct TLabelsPtrEqualKey {
    bool operator()(const TLabelsPtr& l, const TLabelsPtr& r) {
        return *l == *r;
    }
};

// A ht for storing labels to reduce duplication.
// If already existing labels are received, a pointer from the pool is used instead
class TLabelsPool {
public:
    TLabelsPool() = default;

    TLabelsPtr GetOrConstruct(NWithMemoryInfo::TLabels* labels) {
        auto g = Guard(Lock_);

        TLabelsPtr labelsPtr = TLabelsPtr{labels};
        auto [labelsIt, _] = LabelsSet_.emplace(labelsPtr);

        return *labelsIt;
    }

    void Erase(const TVector<const TLabelsPtr>& labelsPtrs) {
        auto g = Guard(Lock_);

        for (const TLabelsPtr& key: labelsPtrs) {
            // 3 refs: LabelsSet_, Aggregators_, keysToRemove
            if (static_cast<ui64>(key.ReferenceCounter()->Val()) <= 3) {
                LabelsSet_.erase(key);
            }
        }
    }

private:
    TAdaptiveLock Lock_;
    THashSet<TLabelsPtr, TLabelsPtrHasher, TLabelsPtrEqualKey> LabelsSet_;
};

constexpr ui64 SkippedMetricTypeToIndex(EMetricType type) {
    switch (type) {
        case EMetricType::COUNTER:
            return 0;
        case EMetricType::RATE:
            return 1;
        case EMetricType::HIST_RATE:
            return 2;
        default:
            ythrow yexception() << "Unknown metric type";
    }
}

using TMetricDataPtr = TSimpleSharedPtr<TMetricData>;
using TLabelsToMetricData = THashMap<TLabelsPtr, TMetricDataPtr>;
using TSkippedMetricsCounters = std::array<std::atomic<ui64>, 3>;

class IPointsProcessor: public TAtomicRefCount<IPointsProcessor> {
public:
    virtual void Process(TLabelsToMetricData&& activeMetricsPoints, TLabelsToMetricData&& deferredMetricsPoints) = 0;

    virtual ~IPointsProcessor() = default;
};

using TPointsProcessorPtr = TIntrusivePtr<IPointsProcessor>;

/**
 * Sort all metrics' points into two groups: active points, that are inside a current interval, and deferred points,
 * that should be processed when a new interval starts
 */
class TSortingMetricsConsumer final: public IStorageMetricsConsumer {
public:
    TSortingMetricsConsumer(TPointsProcessorPtr actor, TLabelsPool& labelsPool, TSkippedMetricsCounters& skippedMetricsCounters,
                            TInstant intervalStart, TDuration intervalDuration)
        : Actor_{actor}
        , IntervalStart_{intervalStart}
        , IntervalDuration_{intervalDuration}
        , LabelsPool_{labelsPool}
        , SkippedMetricsCounters_{skippedMetricsCounters}
    {}

private:
    void OnStreamBegin() override {}
    void OnStreamEnd() override {}

    void OnCommonTime(TInstant time) override {
        // TODO: OnCommonTime could be called after all metrics
        CommonTime_ = time;
    }

    void OnMetricBegin(EMetricType type) override {
        if (type == EMetricType::UNKNOWN) {
            ythrow yexception() << "Metric type is not set";
        }

        if (type != EMetricType::GAUGE &&
            type != EMetricType::IGAUGE &&
            type != EMetricType::DSUMMARY &&
            type != EMetricType::HIST)
        {
            SkippedMetricsCounters_[SkippedMetricTypeToIndex(type)]++;
            NotAggregatable_ = true;
        }

        IsInMetricState_ = true;
        Type_ = type;
    }

    void OnMetricEnd() override {
        Labels_.Reset();

        ActiveTimeSeries_ = nullptr;
        DeferredTimeSeries_ = nullptr;
        IsInMetricState_ = false;
        NotAggregatable_ = false;
    }

    void OnLabelsBegin() override {
        if (NotAggregatable_) {
            return;
        }

        Labels_.Reset(new NWithMemoryInfo::TLabels());
    }
    void OnLabelsEnd() override {
        if (NotAggregatable_) {
            return;
        }

        if (IsInMetricState_) {
            for (auto&& label: CommonLabels_) {
                Labels_->Add(label);
            }
            LabelsPtr_ = LabelsPool_.GetOrConstruct(Labels_.Release());
        }
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        if (NotAggregatable_) {
            return;
        }

        if (IsInMetricState_) {
            Labels_->Add(name, value);
        } else {
            CommonLabels_.Add(name, value);
        }
    }

    template <typename TValue>
    void OnValue(TInstant time, TValue value) {
        if (NotAggregatable_) {
            return;
        }

        time = ComputeTime(Type_, time, CommonTime_);

        TMetricTimeSeries* timeSeries = ChooseTimeSeries(time);
        if (timeSeries) {
            timeSeries->Add(time, value);
        }
    }

    void OnDouble(TInstant time, double value) override {
        OnValue(time, value);
    }
    void OnInt64(TInstant time, i64 value) override {
        OnValue(time, value);
    }
    void OnUint64(TInstant time, ui64 value) override {
        OnValue(time, value);
    }
    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void Flush() override {
        try {
            Actor_->Process(std::move(ActiveMetricsPoints_), std::move(DeferredMetricsPoints_));
        } catch (...) {
            const TString& msg = CurrentExceptionMessage();

            SA_LOG(ERROR) << "failed to process data: " << msg;
        }
    }

    TMetricTimeSeries* ChooseTimeSeries(TInstant time) {
        TInstant intervalStart = IntervalStart_;
        TInstant intervalEnd = intervalStart + IntervalDuration_;
        TInstant nextIntervalEnd = intervalEnd + IntervalDuration_;

        bool isInsideNextInterval = (time >= intervalEnd) && (time < nextIntervalEnd);
        bool isInsideCurrentInterval = (time >= intervalStart) && (time < intervalEnd);

        if (isInsideCurrentInterval) {
            if (!ActiveTimeSeries_) {
                ActiveTimeSeries_ = CreateTimeSeries(ActiveMetricsPoints_);
            }

            return ActiveTimeSeries_;
        }

        if (isInsideNextInterval) {
            if (!DeferredTimeSeries_) {
                DeferredTimeSeries_ = CreateTimeSeries(DeferredMetricsPoints_);
            }

            return DeferredTimeSeries_;
        }

        // TODO: log labels?
        SA_LOG(WARN) << "writing outside of an interval is not allowed"
                     << " (interval: [" << intervalStart << "; " << intervalEnd << ")."
                     << " point time: " << time << ")";

        return nullptr;
    }

    TMetricTimeSeries* CreateTimeSeries(TLabelsToMetricData& labelsToMetricData) {
        auto[it, isInserted] = labelsToMetricData.emplace(LabelsPtr_, TMetricDataPtr(new TMetricData{ {}, Type_ }));
        return &it->second->Series;
    }

private:
    TPointsProcessorPtr Actor_;

    TInstant IntervalStart_;
    TDuration IntervalDuration_;

    TLabelsPool& LabelsPool_;
    TSkippedMetricsCounters& SkippedMetricsCounters_;

    TLabelsToMetricData ActiveMetricsPoints_;
    TLabelsToMetricData DeferredMetricsPoints_;

    EMetricType Type_;
    THolder<NWithMemoryInfo::TLabels> Labels_ = nullptr;
    NWithMemoryInfo::TLabels CommonLabels_;
    TInstant CommonTime_{TInstant::Zero()};
    TLabelsPtr LabelsPtr_;
    bool IsInMetricState_ = false;
    bool NotAggregatable_ = false;

    TMetricTimeSeries* ActiveTimeSeries_ = nullptr;
    TMetricTimeSeries* DeferredTimeSeries_ = nullptr;
};

class TAggregatingMemStorage final: public TMemStorage {
public:
    ~TAggregatingMemStorage() {
        try {
            Stop();
        } catch(...) {
            const TString& msg = CurrentExceptionMessage();
            SA_LOG(ERROR) << "failed to stop a storage properly: " << msg;
        }
    }

    TAggregatingMemStorage(const TStorageShardId& shardId,
                           const TBytes perShardMemoryLimit, const ui32 maxChunks,
                           const TOffsetsSettings& offsetsSettings,
                           TMemoryUsageInfoPtr totalMemoryUsageInfo,
                           TTimerDispatcherPtr timerDispatcher,
                           NActor::TExecutor* actorsExecutor,
                           IStorageUpdateListener* listener)
        // TODO: add a listener for aggregating mem storage. What to log?
        : TMemStorage{shardId, perShardMemoryLimit, maxChunks, offsetsSettings, totalMemoryUsageInfo, listener}
        , Actor_{new TStorageActor(actorsExecutor, this,
                                   timerDispatcher->CurrentIntervalStart(), timerDispatcher->DispatchInterval())}
    {
        Y_ENSURE(timerDispatcher, "no timer dispatcher was given");

        RegisterTimerFunction(timerDispatcher);
    }

private:
    // to store aggregators with different inner types inside one HashMap
    class IMetricAggregator: public TSimpleRefCount<IMetricAggregator> {
    public:
        virtual ~IMetricAggregator() = default;
        virtual void Add(const TPoint& point) = 0;
        virtual bool Empty() const = 0;
    };
    using IMetricAggregatorPtr = TIntrusivePtr<IMetricAggregator>;

    // TODO: add a listener and/or logs for aggregation
    template <typename T>
    class TMetricAggregator final: public IMetricAggregator {
    public:
        void Add(const TPoint& point) override {
            T value = ExtractValueFromPoint(point);

            ++Aggregate_.Count;
            if constexpr (std::is_same_v<T, IHistogramSnapshot*>) {
                if (value) {
                    Aggregate_.Histogram.Add(*value);
                }
            } else if constexpr (std::is_same_v<T, ISummaryDoubleSnapshot*>) {
                if (value) {
                    Aggregate_.Summary.Merge(value);
                }
            } else {
                // TODO: log an overflow
                Aggregate_.Sum += value;
                // TODO:
                /* Aggregate_.Avg = Aggregate_.Sum / Aggregate_.Count; */
                /* Aggregate_.Min = Min(Min, value); */
                /* Aggregate_.Max = Max(Max, value); */
            }
        }

        bool Empty() const override {
            return Aggregate_.Count == 0;
        }

        TAggregate<T> ExtractAggregate() {
            TAggregate<T> aggregateValue = Aggregate_;
            Reset();

            return aggregateValue;
        }

    private:
        T ExtractValueFromPoint(const TPoint& point) {
            if constexpr (std::is_same_v<T, double>) {
                return point.GetValue().AsDouble();
            } else if constexpr (std::is_same_v<T, ui64>) {
                return point.GetValue().AsUint64();
            } else if constexpr (std::is_same_v<T, i64>) {
                return point.GetValue().AsInt64();
            } else if constexpr (std::is_same_v<T, IHistogramSnapshot*>) {
                return point.GetValue().AsHistogram();
            } else if constexpr (std::is_same_v<T, ISummaryDoubleSnapshot*>) {
                return point.GetValue().AsSummaryDouble();
            } else {
                static_assert(TDependentFalse<T>);
            }
        }

        void Reset() {
            Aggregate_.Count = 0;

            if constexpr (std::is_same_v<T, IHistogramSnapshot*>) {
                Aggregate_.Histogram.Reset();
            } else if constexpr (std::is_same_v<T, ISummaryDoubleSnapshot*>) {
                Aggregate_.Summary.Reset();
            } else {
                Aggregate_.Sum = 0;
                // TODO:
                /* Aggregate_.Avg = 0; */
                /* Aggregate_.Min = 0; */
                /* Aggregate_.Max = 0; */
            }
        }

    private:
        TAggregate<T> Aggregate_;
    };

    class TStorageActor: public NActor::TActor<TStorageActor>, public IPointsProcessor {
    public:
        TStorageActor(NActor::TExecutor* executor, TAggregatingMemStorage* storage,
                      TInstant initIntervalStart, TDuration aggregationInterval)
            : NActor::TActor<TStorageActor>{executor}
            , Storage_{storage}
            , AggregationInterval_{aggregationInterval}
        {
            SetCurrentIntervalStart(initIntervalStart);
            for (size_t i = 0; i != SkippedMetricsCounters_.size(); ++i) {
                SkippedMetricsCounters_[i] = 0;
            }
        }

        void Process(TLabelsToMetricData&& activeMetricsPoints, TLabelsToMetricData&& deferredMetricsPoints) override {
            bool atLeastOneSuccessful = false;

            if (!activeMetricsPoints.empty()) {
                try {
                    AggregationTasks_.Enqueue(TTask{std::move(activeMetricsPoints)});
                    atLeastOneSuccessful = true;
                } catch (...) {
                    auto currentInterval = GetCurrentIntervalStart();
                    SA_LOG(ERROR) << "Failed to enqueue the current interval's points"
                                  << " [" << currentInterval << "; " << currentInterval + AggregationInterval_
                                  << "): " << CurrentExceptionMessage();
                }
            }

            if (!deferredMetricsPoints.empty()) {
                try {
                    DeferredTasks_.Enqueue(TTask{std::move(deferredMetricsPoints)});
                    atLeastOneSuccessful = true;
                } catch (...) {
                    auto currentInterval = GetCurrentIntervalStart();
                    SA_LOG(ERROR) << "Failed to enqueue the next interval's points"
                                  << " [" << currentInterval << "; " << currentInterval + AggregationInterval_
                                  << "): " << CurrentExceptionMessage();
                }
            }

            if (atLeastOneSuccessful) {
                Schedule();
            }
        }

        void Flush(TInstant newIntervalStart) {
            FlushTasks_.Enqueue(newIntervalStart);
            Schedule();
        }

        void Act(NActor::TDefaultTag) {
            if (!Storage_->RunningTasksCountdown_.TryInc()) {
                // Counter is already stopped
                return;
            }

            Y_SCOPE_EXIT(this) {
                Storage_->RunningTasksCountdown_.Dec();
            };

            TInstant oldIntervalStart = GetCurrentIntervalStart();
            TInstant newStart;

            if (FlushTasks_.Dequeue(&newStart)) {
                SetCurrentIntervalStart(newStart);

                // Closing interval
                TVector<TTask> aggrTasks;
                AggregationTasks_.DequeueAll(&aggrTasks);
                ProcessTasks(std::move(aggrTasks));

                FlushAggregatorsSafe(oldIntervalStart);

                // New interval
                TVector<TTask> deferredTasks;
                DeferredTasks_.DequeueAll(&deferredTasks);
                ProcessTasks(std::move(deferredTasks));
            } else {
                TVector<TTask> aggrTasks;
                AggregationTasks_.DequeueAll(&aggrTasks);

                ProcessTasks(std::move(aggrTasks));
            }
        }

        IStorageMetricsConsumerPtr CreateConsumer() {
            return MakeHolder<TSortingMetricsConsumer>(
                    this,
                    LabelsPool_,
                    SkippedMetricsCounters_,
                    GetCurrentIntervalStart(),
                    AggregationInterval_);
        }

    private:
        struct TTask: public TMoveOnly {
            TLabelsToMetricData LabelsToMetricData_;

            TTask(TLabelsToMetricData&& labelsToMetricData)
                : LabelsToMetricData_{std::move(labelsToMetricData)}
            {}
        };

        template <typename T>
        using TMetricAggregatorPtr = TIntrusivePtr<TMetricAggregator<T>>;

        template <typename T>
        TMetricAggregator<T>* GetOrCreateAggregator(const TLabelsPtr labels, const EMetricType& type) {
            TMetricAggregator<T>* aggregator;
            auto it = Aggregators_.find(labels);

            if (it != Aggregators_.end()) {
                aggregator = reinterpret_cast<TMetricAggregator<T>*>(it->second.Aggregator.Get());
            } else {
                TIntrusivePtr<TMetricAggregator<T>> newAggregator = new TMetricAggregator<T>();
                auto [it, isInserted] = Aggregators_.emplace(labels, TAggregatorData{type, newAggregator});
                Y_ENSURE(isInserted, "failed to create an aggregator for metric " << *labels);

                aggregator = newAggregator.Get();
            }

            return aggregator;
        }

        template <typename T>
        void PassPointsToAggregator(TMetricAggregator<T>* aggregator, const TMetricTimeSeries& series) {
            for (size_t i = 0; i != series.Size(); ++i) {
                aggregator->Add(series[i]);
            }
        }

        void WriteAggregate(TMetricTimeSeries& timeSeries, EMetricType type, TInstant time, IMetricAggregatorPtr aggregator) {
            // TODO: remove this check when the new processing will be turned on by default
            if (type == EMetricType::RATE || type == EMetricType::HIST_RATE) {
                time = TInstant::Zero();
            }

            switch (type) {
                // TODO: Uncomment after the fix. See https://nda.ya.ru/t/50N0Rfqt3VvZTf for more info

                /* case EMetricType::RATE: */
                /* case EMetricType::COUNTER: { */
                /*     auto agg = reinterpret_cast<TMetricAggregator<ui64>*>(aggregator.Get()); */
                /*     timeSeries.Add(time, agg->ExtractAggregate().Sum); */
                /*     break; */
                /* } */
                /* case EMetricType::HIST_RATE: { */
                /*     auto agg = reinterpret_cast<TMetricAggregator<IHistogramSnapshot*>*>(aggregator.Get()); */
                /*     TIntrusivePtr<IHistogramSnapshot> histogramPtr = */
                /*         dynamic_cast<IHistogramSnapshot*>(new TMutableHistogramSnapshot(std::move(agg->ExtractAggregate().Histogram))); */
                /*     timeSeries.Add(time, histogramPtr.Get()); */
                /*     break; */
                /* } */
                /* case EMetricType::UNKNOWN: */
                /*     ythrow yexception() << "Unknown aggregate type"; */
                case EMetricType::DSUMMARY: {
                    auto agg = reinterpret_cast<TMetricAggregator<ISummaryDoubleSnapshot*>*>(aggregator.Get());
                    auto summaryPtr = MakeIntrusive<TMutableSummaryDouble>(std::move(agg->ExtractAggregate().Summary));
                    timeSeries.Add(time, summaryPtr.Get());
                    break;
                }
                case EMetricType::HIST: {
                    auto agg = reinterpret_cast<TMetricAggregator<IHistogramSnapshot*>*>(aggregator.Get());
                    auto histPtr = MakeIntrusive<TMutableHistogramSnapshot>(std::move(agg->ExtractAggregate().Histogram));
                    timeSeries.Add(time, histPtr.Get());
                    break;
                }
                case EMetricType::IGAUGE: {
                    auto agg = reinterpret_cast<TMetricAggregator<i64>*>(aggregator.Get());
                    timeSeries.Add(time, agg->ExtractAggregate().Sum);
                    break;
                }
                case EMetricType::GAUGE: {
                    auto agg = reinterpret_cast<TMetricAggregator<double>*>(aggregator.Get());
                    timeSeries.Add(time, agg->ExtractAggregate().Sum);
                    break;
                }
                default:
                    ythrow yexception() << "unsupported metric type for aggregation: " << type << "."
                                        << " See https://nda.ya.ru/t/50N0Rfqt3VvZTf for more info";
            }
        }

        void FlushAggregatorsSafe(TInstant time) {
            // TODO: create a metric for dropped values (self monitoring)
            try {
                FlushAggregators(time);
            } catch (...) {
                const TString& msg = CurrentExceptionMessage();
                SA_LOG(ERROR) << "failed to write aggregators data to the storage: " << msg;
                SA_LOG(ERROR) << "destroying aggregators after an error";

                try {
                    Aggregators_.clear();
                } catch (...) {
                    const TString& msg = CurrentExceptionMessage();
                    ythrow yexception() << "failed to destroy aggregators: " << msg;
                }
            }
        }

        void LogSkippedMetricsNumbers() {
            ui64 skippedCounter = SkippedMetricsCounters_[SkippedMetricTypeToIndex(EMetricType::COUNTER)].exchange(0);
            ui64 skippedRate = SkippedMetricsCounters_[SkippedMetricTypeToIndex(EMetricType::RATE)].exchange(0);
            ui64 skippedHistRate = SkippedMetricsCounters_[SkippedMetricTypeToIndex(EMetricType::HIST_RATE)].exchange(0);

            ui64 skippedTotal = skippedCounter + skippedRate + skippedHistRate;

            if (skippedTotal > 0) {
                TString info = TStringBuilder() << "skipped " << skippedTotal << " metrics with types: ";

                if (skippedCounter) {
                    info += TStringBuilder() << "COUNTER=" << skippedCounter << ", ";
                }
                if (skippedRate) {
                    info += TStringBuilder() << "RATE=" << skippedRate << ", ";
                }
                if (skippedHistRate) {
                    info += TStringBuilder() << "HIST_RATE=" << skippedHistRate << ", ";
                }

                info += "see https://nda.ya.ru/t/50N0Rfqt3VvZTf";

                SA_LOG(WARN) << info;
            }
        }

        void FlushAggregators(TInstant time) {
            TVector<const TLabelsPtr> keysToRemove;
            TChunkPtr chunk = MakeIntrusive<TChunk>();

            for (auto& [labels, aggregatorData]: Aggregators_) {
                EMetricType& type = aggregatorData.Type;
                IMetricAggregatorPtr& aggregator = aggregatorData.Aggregator;

                if (aggregator->Empty()) {
                    keysToRemove.emplace_back(labels);
                } else {
                    TMetricData data;
                    data.Type = type;
                    WriteAggregate(data.Series, type, time, aggregator);

                    chunk->AddMetricData(NWithMemoryInfo::TLabels{*labels}, std::move(data));
                }
            }

            // TODO: get rid of it when the new processing will be fixed
            LogSkippedMetricsNumbers();

            if (chunk->Size() > 0) {
                Storage_->TryWriteChunk(std::move(chunk));
            }

            LabelsPool_.Erase(keysToRemove);
            // TODO: Optimization: there's a slight chance that an aggregator will be destroyed and created right after
            // Destroy it under a labels lock?
            for (const TLabelsPtr& key: keysToRemove) {
                Aggregators_.erase(key);
            }
        }

        void AggregatePoints(const TLabelsPtr labels, const EMetricType& type, const TMetricTimeSeries& series) {
            switch (type) {
                // TODO: Uncomment after the fix. See https://nda.ya.ru/t/50N0Rfqt3VvZTf for more info

                /* case EMetricType::RATE: */
                /* case EMetricType::COUNTER: { */
                /*     PassPointsToAggregator(GetOrCreateAggregator<ui64>(labels, type), series); */
                /*     break; */
                /* } */
                /* case EMetricType::HIST_RATE: { */
                /*     PassPointsToAggregator(GetOrCreateAggregator<IHistogramSnapshot*>(labels, type), series); */
                /*     break; */
                /* } */
                /* case EMetricType::UNKNOWN: */
                /*     ythrow yexception() << "Unknown point type"; */
                case EMetricType::DSUMMARY: {
                    PassPointsToAggregator(GetOrCreateAggregator<ISummaryDoubleSnapshot*>(labels, type), series);
                    break;
                }
                case EMetricType::HIST: {
                    PassPointsToAggregator(GetOrCreateAggregator<IHistogramSnapshot*>(labels, type), series);
                    break;
                }
                case EMetricType::IGAUGE: {
                    PassPointsToAggregator(GetOrCreateAggregator<i64>(labels, type), series);
                    break;
                }
                case EMetricType::GAUGE: {
                    PassPointsToAggregator(GetOrCreateAggregator<double>(labels, type), series);
                    break;
                }
                default:
                    ythrow yexception() << "unsupported metric type for aggregation: " << type << "."
                                        << " See https://nda.ya.ru/t/50N0Rfqt3VvZTf for more info";
            }
        }

        void ProcessTasks(TVector<TTask>&& tasks) {
            try {
                for (TTask& task: tasks) {
                    for (auto& [labels, metricData]: task.LabelsToMetricData_) {
                        AggregatePoints(labels, metricData->Type, metricData->Series);
                    }
                }
            } catch (...) {
                const TString& msg = CurrentExceptionMessage();
                ythrow yexception() << "Failed to process aggregation tasks: " << msg;
            }
        }

        TInstant GetCurrentIntervalStart() const {
            return TInstant::FromValue(CurrentIntervalStart_.load(std::memory_order_relaxed));
        }

        void SetCurrentIntervalStart(TInstant time) {
            CurrentIntervalStart_.store(time.GetValue(), std::memory_order_relaxed);
        }

    private:
        struct TAggregatorData {
            EMetricType Type;
            IMetricAggregatorPtr Aggregator;
        };
        THashMap<TLabelsPtr, TAggregatorData, TLabelsPtrHasher, TLabelsPtrEqualKey> Aggregators_;

        TLabelsPool LabelsPool_;

        TAggregatingMemStorage* Storage_;

        std::atomic_uint64_t CurrentIntervalStart_;
        TDuration AggregationInterval_;

        TLockFreeQueue<TTask> AggregationTasks_;
        TLockFreeQueue<TTask> DeferredTasks_;
        TLockFreeQueue<TInstant> FlushTasks_;

        TSkippedMetricsCounters SkippedMetricsCounters_;
    };

    void RegisterTimerFunction(const TTimerDispatcherPtr timerDispatcher) {
        try {
            DispatcherFunctionHolder_ =
                timerDispatcher->RegisterWithOwning([this](TInstant nextIntervalStart) {
                    if (!RunningTasksCountdown_.TryInc()) {
                        // Counter is already stopped
                        return;
                    }

                    Y_SCOPE_EXIT(this) {
                        RunningTasksCountdown_.Dec();
                    };

                    Actor_->Flush(nextIntervalStart);
                });

            SA_LOG(DEBUG) << "started an aggregation task";
        } catch (...) {
            ythrow yexception() << "failed to create a timer function";
        }
    }

    void Stop() {
        DispatcherFunctionHolder_.Release();
        RunningTasksCountdown_.Stop();

        while (!RunningTasksCountdown_.Await(TDuration::Seconds(1))) {
            SA_LOG(DEBUG) << "waiting for aggregation tasks completion";
        }
    }

    IStorageMetricsConsumerPtr CreateConsumer(TInstant) override {
            return Actor_->CreateConsumer();
    }

private:
    TTimerDispatcher::TFunctionHolder DispatcherFunctionHolder_;
    TCountdownEvent RunningTasksCountdown_;

    TIntrusivePtr<TStorageActor> Actor_;
};

} // namespace

IStoragePtr CreateAggregatingMemStorage(
        const TStorageShardId& shardId,
        const TBytes perShardMemoryLimit,
        const ui32 maxChunks,
        const TOffsetsSettings& offsetsSettings,
        TMemoryUsageInfoPtr totalMemoryUsageInfo,
        TTimerDispatcherPtr timerDispatcher,
        NActor::TExecutor* actorsExecutor,
        IStorageUpdateListener* listener)
{
    listener = listener != nullptr ? listener : TDummyStorageListener::Get();
    return new TAggregatingMemStorage(
        shardId, perShardMemoryLimit, maxChunks, offsetsSettings,
        totalMemoryUsageInfo, timerDispatcher, actorsExecutor, listener);
}

} // namespace NSolomon::NAgent {
