#include "common.h"

#include <util/generic/strbuf.h>
#include <util/stream/output.h>
#include <util/string/cast.h>

namespace NSolomon::NAgent {
    using TLabel = TAgentLabel;
    using TLabels = TAgentLabels;

    extern const THashMap<TStringBuf, ui64> POSTFIX_TO_BASE = {
        {"B", 1},
        {"KiB", 1024},
        {"MiB", 1024 * 1024},
        {"GiB", 1024 * 1024 * 1024}
    };

    TBytes ParseHumanReadableSize(const TString& sizeString) {
        TBytes size = 0;

        try {
            auto nonDigitIt = std::find_if_not(sizeString.begin(), sizeString.end(), IsAsciiDigit<char>);
            if (nonDigitIt == sizeString.end()) {
                return FromString<ui64>(sizeString);
            }

            TStringBuf postfix = TStringBuf{sizeString}.Last(sizeString.end() - nonDigitIt);

            auto it = POSTFIX_TO_BASE.find(postfix);
            Y_ENSURE(it != POSTFIX_TO_BASE.end());

            size = FromString<ui64>(TStringBuf{sizeString}.Chop(postfix.size()));
            size *= it->second;
        } catch (...) {
            ythrow yexception() << "unknown size format: \"" << sizeString << "\". Supported formats: 1B, 1KiB, 1MiB, 1GiB";
        }

        return size;
    }

    TInstant ComputeTime(NMonitoring::EMetricType type, TInstant time, TInstant commonTime, TInstant defaultTs) {
        TInstant originalTime = time ? time : commonTime;

        // TODO: remove this check when the new processing will be turned on by default
        if (type == NMonitoring::EMetricType::RATE || type == NMonitoring::EMetricType::HIST_RATE) {
            return originalTime;
        } else {
            if (originalTime) {
                return originalTime;
            }

            if (defaultTs) {
                return defaultTs;
            }

            return TInstant::Now();
        }
    }

} // namespace NSolomon::NAgent

template <>
void Out<NSolomon::NAgent::TStorageShardId>(IOutputStream& out, const NSolomon::NAgent::TStorageShardId& shardId) {
    out << TStringBuf("Shard{project=") << shardId.Project
        << TStringBuf("; service=") << shardId.Service << TStringBuf("}");
}

Y_MONLIB_DEFINE_LABELS_OUT(NSolomon::NAgent::NWithMemoryInfo::TLabels);
