#pragma once

#include <solomon/agent/misc/labels.h>
#include <solomon/agent/misc/mutable_histogram.h>

#include <library/cpp/monlib/metrics/metric_type.h>
#include <library/cpp/monlib/metrics/metric_value.h>

#include <util/digest/multi.h>
#include <util/generic/hash.h>
#include <util/generic/strbuf.h>

#include <algorithm>

namespace NSolomon::NAgent {
    using TBytes = ui64;
    constexpr TBytes MAX_STORAGE_LIMIT {1ull * 1024 * 1024 * 1024 * 1024}; // 1 TiB

    extern const THashMap<TStringBuf, ui64> POSTFIX_TO_BASE;

    struct TStorageShardId {
        TString Project;
        TString Service;

        bool operator==(const TStorageShardId& other) const {
            return Project == other.Project && Service == other.Service;
        }

        ui64 Hash() const {
            return MultiHash(Project, Service);
        }
    };

    struct TOffsetsSettings {
        TDuration SoftTTL{TDuration::Minutes(30)};
        TDuration HardTTL{TDuration::Hours(5)};
        TDuration WatchInterval{TDuration::Minutes(5)};
    };

    TBytes ParseHumanReadableSize(const TString& sizeString);
    TString SizeToString(TBytes bytes);

    namespace NWithMemoryInfo {
        class TLabel: public TAgentLabel {
        public:
            ui64 MemorySizeBytes() const {
                ui64 size = sizeof(*this);

                // Is a string SSOptimized? More on the topic:
                // https://stackoverflow.com/questions/21694302/what-are-the-mechanics-of-short-string-optimization-in-libc/21710033#21710033
                // https://stackoverflow.com/questions/21694302/what-are-the-mechanics-of-short-string-optimization-in-libc/21696699#21696699
                if (NameStr().__is_long()) {
                    size += NameStr().capacity();
                } // otherwise, its size is already counted inside sizeof()

                if (ValueStr().__is_long()) {
                    size += ValueStr().capacity();
                }

                return size;
            }
        };

        class TLabels: public TAgentLabels {
        public:
            ui64 MemorySizeBytes() const {
                ui64 bytes = sizeof(*this);

                for (const auto& label: *this) {
                    bytes += static_cast<const TLabel&>(label).MemorySizeBytes();
                }

                bytes += (capacity() - size()) * sizeof(TAgentLabels::value_type);

                return bytes;
            }
        };

        class TMetricTimeSeries: public NMonitoring::TMetricTimeSeries {
            TBytes HistogramSizeFallback_ = sizeof(NMonitoring::TBucketBounds) + sizeof(NMonitoring::TBucketValues);
            TBytes HistogramPointSizeFallback_ = sizeof(NMonitoring::TBucketBound) + sizeof(NMonitoring::TBucketValue);

        public:
            TBytes MemorySizeBytes() const {
                TBytes size = sizeof(*this) + Capacity() * sizeof(TPoint);

                if (GetValueType() == NMonitoring::EMetricValueType::SUMMARY) {
                    for (size_t i = 0; i != Size(); ++i) {
                        NMonitoring::ISummaryDoubleSnapshot* summary =(*this)[i].GetValue().AsSummaryDouble();
                        size += summary->MemorySizeBytes();
                    }
                } else if (GetValueType() == NMonitoring::EMetricValueType::HISTOGRAM) {
                    for (size_t i = 0; i != Size(); ++i) {
                        NMonitoring::IHistogramSnapshot* hist = (*this)[i].GetValue().AsHistogram();

                        if (auto downcasted = dynamic_cast<NMonitoring::TExplicitHistogramSnapshot*>(hist)) {
                            size += downcasted->MemorySizeBytes();
                        } else if (auto downcasted = dynamic_cast<NMonitoring::TLinearHistogramSnapshot*>(hist)) {
                            size += downcasted->MemorySizeBytes();
                        } else if (auto downcasted = dynamic_cast<NMonitoring::TExponentialHistogramSnapshot*>(hist)) {
                            size += downcasted->MemorySizeBytes();
                        } else if (auto downcasted = dynamic_cast<TMutableHistogramSnapshot*>(hist)) {
                            size += downcasted->MemorySizeBytes();
                        } else {
                            Y_VERIFY_DEBUG(false, "No other types of a histogram snapshot other than"
                                                  " NMonitoring::THistogramSnapshot can be used");

                            // Actual class of a snapshot is unknown. Will use a rough estimation

                            // Histogram own size
                            size += HistogramSizeFallback_;
                            // Sum size of all points
                            size += hist->Count() * HistogramPointSizeFallback_;
                        }
                    }
                } else if (GetValueType() == NMonitoring::EMetricValueType::LOGHISTOGRAM) {
                    for (size_t i = 0; i != Size(); ++i) {
                        NMonitoring::TLogHistogramSnapshot* logHist =(*this)[i].GetValue().AsLogHistogram();
                        size += logHist->MemorySizeBytes();
                    }
                }

                return size;
            }
        };

    } // namespace NWithMemoryInfo

    TInstant ComputeTime(NMonitoring::EMetricType type, TInstant time, TInstant commonTime,
                         TInstant defaultTs = TInstant::Zero());

} // namespace NSolomon::NAgent


template<>
struct THash<NSolomon::NAgent::TStorageShardId> {
    inline ui64 operator()(const NSolomon::NAgent::TStorageShardId& shardId) const {
        return shardId.Hash();
    }
};

template <>
struct THash<NSolomon::NAgent::NWithMemoryInfo::TLabel> {
    inline size_t operator()(const NSolomon::NAgent::NWithMemoryInfo::TLabel& label) const {
        return label.Hash();
    }
};

template <>
struct THash<NSolomon::NAgent::NWithMemoryInfo::TLabels> {
    inline size_t operator()(const NSolomon::NAgent::NWithMemoryInfo::TLabels& labels) const {
        return labels.Hash();
    }
};
