#include "mem_storage.h"

#include <solomon/agent/misc/labels_merge.h>

#include <library/cpp/monlib/metrics/histogram_collector.h>

#include <util/system/spinlock.h>
#include <util/generic/algorithm.h>
#include <util/generic/deque.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/string/strip.h>

using NMonitoring::IMetricConsumer;
using NMonitoring::EMetricType;
using NMonitoring::EMetricValueType;
using NMonitoring::TMetricValue;
using NMonitoring::TMetricTimeSeries;


namespace NSolomon {
namespace NAgent {
namespace {

constexpr auto CHUNK_WRITE_LIMIT = TSeqNo::MaxMetricOffset();

template <typename TProtoLabels>
void CopyLabels(const TProtoLabels& src, TLabels* dst) {
    for (const auto& l: src) {
        dst->Add(l.GetName(), l.GetValue());
    }
}

void WriteLabels(IMetricConsumer* c, const TLabels& labels) {
    c->OnLabelsBegin();
    for (const auto& l: labels) {
        c->OnLabel(TString{l.Name()}, TString{l.Value()});
    }
    c->OnLabelsEnd();
};

void WriteValues(IMetricConsumer* c, const TMetricTimeSeries& data) {
    std::function<void(TInstant, EMetricValueType, TMetricValue)> consume;
    switch (data.GetValueType()) {
    case EMetricValueType::DOUBLE:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnDouble(time, val.AsDouble());
        };
        break;

    case EMetricValueType::INT64:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnInt64(time, val.AsInt64());
        };
        break;

    case EMetricValueType::UINT64:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnUint64(time, val.AsUint64());
        };
        break;

    case EMetricValueType::HISTOGRAM:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnHistogram(time, val.AsHistogram());
        };
        break;

    case EMetricValueType::SUMMARY:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnSummaryDouble(time, val.AsSummaryDouble());
        };
        break;

    case EMetricValueType::LOGHISTOGRAM:
        consume = [c] (TInstant time, EMetricValueType, TMetricValue val) {
            c->OnLogHistogram(time, val.AsLogHistogram());
        };
        break;

    case EMetricValueType::UNKNOWN:
        ythrow yexception() << "value type is not set";
    }

    data.ForEach(consume);
}

} // namespace


///////////////////////////////////////////////////////////////////////////////
// TMemStorage
// TODO: merge with same timestamp
///////////////////////////////////////////////////////////////////////////////
TMemStorage::TMemStorage(
        const TStorageShardId& shardId,
        const TBytes perShardMemoryLimit,
        const ui32 maxChunks,
        const TOffsetsSettings& offsetsSettings,
        TMemoryUsageInfoPtr totalMemoryUsageInfo,
        IStorageUpdateListener* listener)
    : ShardId_{shardId}
    , CurrentShardMemoryUsage_{sizeof(TMemStorage)}
    , MemoryLimit_{perShardMemoryLimit}
    , MaxChunks_{maxChunks}
    , TotalMemoryUsageInfo_{totalMemoryUsageInfo}
    , Offsets_{offsetsSettings.SoftTTL, offsetsSettings.HardTTL}
    , TimerDispatcher_{offsetsSettings.WatchInterval}
    , UpdateListener_{listener}
{
    try {
        Y_ENSURE(MemoryLimit_ > CurrentShardMemoryUsage_);

        // How much memory can be used to store real data and can be freed to write chunks that are larger than free memory.
        // At this point, CurrentShardMemoryUsage_'s value is the size of TMemStorage, so this memory won't be freed
        DataMemoryLimit_ = MemoryLimit_ - CurrentShardMemoryUsage_;

        Y_ENSURE(TotalMemoryUsageInfo_->TryIncreaseSize(CurrentShardMemoryUsage_));

        UpdateListener_->OnLimitSet(perShardMemoryLimit);
    } catch (...) {
        ythrow yexception() << "cannot create the shard " << ShardId_ << ": out of memory";
    }

    RegisterTimerFunction();
}

void TMemStorage::RegisterTimerFunction() {
    try {
        TimerDispatcher_.RegisterSilently([this](TInstant) {
            Offsets_.CleanupByTTL();
        });

        SA_LOG(DEBUG) << "started an offset TTL task";
    } catch (...) {
        ythrow yexception() << "failed to create a timer function";
    }
}

// -- read member functions -----------------------------------------------

TMetricConstIterator TMemStorage::ReadRawData(const TQuery& query) {
    TChunkPointers rawData;
    TSeqNo offset;

    {
        auto data = Data_.Read();
        if (data->empty()) {
            return TMetricConstIterator{};
        }

        offset = CalculateOffset(*data, query.ConsumerId(), query.Offset());

        auto dataBegin = FindChunk(*data, offset);
        auto dataEnd = std::end(*data);

        if (dataBegin == dataEnd) {
            return TMetricConstIterator{};
        }

        rawData.reserve(std::distance(dataBegin, dataEnd));
        std::copy(dataBegin, dataEnd, std::back_inserter(rawData));
    }

    return TMetricConstIterator{std::move(rawData), offset};
}

void TMemStorage::UpdateDirtyOffset(const TQuery& query, TSeqNo dirtySeqNo) {
    if (query.ConsumerId()) {
        Offsets_.CommitDirty(*query.ConsumerId(), dirtySeqNo);
    }
}

TReadResult TMemStorage::Read(
        const TQuery& query,
        IMetricConsumer* c,
        const TReadOptions&)
{
    auto it = ReadRawData(query);
    if (!it.HasNext()) {
        auto dirtySeqNo = EndSeqNo();
        UpdateDirtyOffset(query, dirtySeqNo);

        return {dirtySeqNo, false, 0};
    }

    ui32 addedCount = 0;
    TLabelsMatcher matcher = query.ToMatcher();

    c->OnStreamBegin();

    while (it.HasNext()) {
        if (addedCount >= query.Limit()) {
            c->OnStreamEnd();

            auto dirtySeqNo = it.ToSeqNo();
            UpdateDirtyOffset(query, dirtySeqNo);

            return {dirtySeqNo, true, addedCount};
        }

        const auto& metricWithCommonLabels = it.Next();

        auto&& commonLabels = metricWithCommonLabels.CommonLabels;
        auto&& metric = metricWithCommonLabels.Metric;
        auto&& labels = metric.first;
        auto&& data = metric.second;

        if (!matcher(labels, &commonLabels)) {
            continue;
        }

        c->OnMetricBegin(data.Type);

        c->OnLabelsBegin();
        PerformOnMergedLabels(labels, commonLabels, [&c](auto&& l) {
            c->OnLabel(l.Name(), l.Value());
        });
        c->OnLabelsEnd();

        WriteValues(c, data.Series);
        c->OnMetricEnd();

        ++addedCount;
    }

    c->OnStreamEnd();

    auto dirtySeqNo = EndSeqNo();
    UpdateDirtyOffset(query, dirtySeqNo);

    return {dirtySeqNo, false, addedCount};
}

TFindResult TMemStorage::Find(
        const TQuery& query,
        IMetricConsumer* c,
        const TFindOptions&)
{
    TQuery newQuery{query};
    newQuery.ConsumerId({}); // because it doesn't make sense to have a consumer offset in a find query
    newQuery.Offset(INITIAL_OFFSET);

    auto it = ReadRawData(newQuery);
    if (!it.HasNext()) {
        auto dirtySeqNo = EndSeqNo();
        UpdateDirtyOffset(query, dirtySeqNo);

        return {dirtySeqNo, false};
    }

    TLabelsMatcher matcher = query.ToMatcher();
    THashSet<TLabels> alreadyAdded;

    c->OnStreamBegin();

    while (it.HasNext()) {
        if (alreadyAdded.size() >= query.Limit()) {
            c->OnStreamEnd();

            auto dirtySeqNo = it.ToSeqNo();
            UpdateDirtyOffset(query, dirtySeqNo);

            return {dirtySeqNo, true};
        }

        auto metricWithCommonLabels = it.Next();

        auto&& commonLabels = metricWithCommonLabels.CommonLabels;
        auto&& metric = metricWithCommonLabels.Metric;
        auto&& labels = metric.first;
        auto&& data = metric.second;

        TLabels mergedLabels = MergeSortedUsualAndCommonLabels(labels, commonLabels);

        if (!matcher(mergedLabels, /*optionalLabels =*/ nullptr) || alreadyAdded.contains(mergedLabels)) {
            continue;
        }

        c->OnMetricBegin(data.Type);
        WriteLabels(c, mergedLabels);
        c->OnMetricEnd();

        alreadyAdded.emplace(mergedLabels);
    }

    c->OnStreamEnd();

    auto dirtySeqNo = EndSeqNo();
    UpdateDirtyOffset(query, dirtySeqNo);

    return {dirtySeqNo, false};
}

class TMemStorage::TMemStorageMetricsConsumer final: public IStorageMetricsConsumer {
public:
    using TMemStoragePtr = TIntrusivePtr<TMemStorage>;

    TMemStorageMetricsConsumer(TMemStoragePtr storage, TInstant defaultTs)
        : Storage_{storage}
        , DefaultTs_{defaultTs}
        , Result_{}
        , Chunk_{Result_.emplace_back(MakeIntrusive<TChunk>())}
    {}

private:
    void OnStreamBegin() override {}
    void OnStreamEnd() override {}

    void Flush() override {
        if (Result_.size() != 0 && Result_[0]->Size() != 0) {
            CommonLabels_.Sort();

            for (auto& chunk: Result_) {
                chunk->CopyCommonLabels(CommonLabels_);
            }

            Storage_->Write(std::move(Result_));
        }

        // TODO: create a state value that states that the object is no longer usable
    }

    void OnCommonTime(TInstant time) override {
        // TODO: OnCommonTime could be called after all metrics
        CommonTime_ = time;
    }

    void OnMetricBegin(EMetricType type) override {
        if (type == EMetricType::UNKNOWN) {
            ythrow yexception() << "Metric type is not set";
        }

        IsInMetricState_ = true;
        MetricData_.Type = type;
    }

    void OnMetricEnd() override {
        // TODO: use labelsPool(hash set)

        Labels_.Sort();
        Chunk_->AddMetricData(std::move(Labels_), std::move(MetricData_));

        IsInMetricState_ = false;
        MetricData_.Type = EMetricType::UNKNOWN;

        Labels_.Clear();
        MetricData_ = {};

        if (Chunk_->Size() == CHUNK_WRITE_LIMIT) {
            AppendChunk();
        }
    }

    void OnLabelsBegin() override {}

    void OnLabelsEnd() override {}

    void OnLabel(TStringBuf name, TStringBuf value) override {
        if (IsInMetricState_) {
            Labels_.Add(name, value);
        } else {
            CommonLabels_.Add(name, value);
        }
    }

    template <typename TValueType>
    void OnValue(TInstant time, TValueType value) {
        time = ComputeTime(MetricData_.Type, time, CommonTime_, DefaultTs_);
        MetricData_.Series.Add(time, value);
    }

    void OnDouble(TInstant time, double value) override {
        OnValue(time, value);
    }

    void OnInt64(TInstant time, i64 value) override {
        OnValue(time, value);
    }

    void OnUint64(TInstant time, ui64 value) override {
        OnValue(time, value);
    }

    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        OnValue(time, snapshot.Get());
    }

    void AppendChunk() {
        Chunk_ = Result_.emplace_back(MakeIntrusive<TChunk>());
    }

private:
    TMemStoragePtr Storage_;
    TInstant DefaultTs_;
    bool IsInMetricState_ = false;

    TLabels Labels_;
    TLabels CommonLabels_;
    TInstant CommonTime_{TInstant::Zero()};

    TMetricData MetricData_;
    TDataChunks Result_;
    TChunkPtr Chunk_{nullptr};
};


// -- write member functions ----------------------------------------------

IStorageMetricsConsumerPtr TMemStorage::CreateConsumer(TInstant defaultTs) {
    return MakeHolder<TMemStorageMetricsConsumer>(this, defaultTs);
}

void TMemStorage::Write(TDataChunks&& chunks) {
    size_t skippedChunks = 0;
    TStringBuilder errorMsg;

    for (auto&& chunk: chunks) {
        try {
            TryWriteChunk(std::move(chunk));
        } catch (...) {
            ++skippedChunks;

            if (skippedChunks <= 5) {
                if (errorMsg.size() != 0) {
                    errorMsg << ". ";
                }

                errorMsg << "#" << skippedChunks << " " << CurrentExceptionMessage();
            }
        }
    }

    if (skippedChunks > 0) {
        TStringBuilder msgPrefix;
        if (skippedChunks == chunks.size()) {
            msgPrefix << "Failed to write all (" << skippedChunks << ") chunks of data";
        } else {
            msgPrefix << "Failed to write " << skippedChunks << " chunks of data out of " << chunks.size();
        }

        ythrow yexception() << msgPrefix << " to " << ShardId_ << " due to errors: " << errorMsg;
    }
}

void TMemStorage::Delete(const TQuery&, const TDeleteOptions&) {
    Y_FAIL("not implemented");
}

void TMemStorage::Commit(const TString& consumerId, TSeqNo seqNo) {
    {
        auto data = Data_.Write();
        if (!Offsets_.Commit(consumerId, seqNo, EndSeqNo())) {
            return;
        }

        RemoveReadData(*data);
    }

    UpdateListener_->OnFetcherUpdated(consumerId, seqNo.ChunkOffset());
}

/**
 * A TSeqNo value of a chunk that will be written _next_
 */
TSeqNo TMemStorage::EndSeqNo() const {
    // Using Data_.back().SeqNo().NextChunk() is not sufficient, because the storage can be empty
    // but EndSeqNo() still shoule be > 0. For example, when all read chunks are deleted, but a new chunk
    // should have a seqno greater than it was in a last chunk before deletion
    return EndSeqNo_;
}

// must be called under read lock
TSeqNo TMemStorage::CalculateOffset(const TDataChunks& data, const TMaybe<TString>& consumerId, TMaybe<TSeqNo> offset) {
    TSeqNo result = data.empty() ? INITIAL_OFFSET : data.front()->SeqNo();

    if (!(consumerId || offset)) {
        return result;
    }

    if (consumerId) {
        offset = Offsets_.GetOrInitConsumerOffset(*consumerId);
    }

    return Max(*offset, result);
}

bool TMemStorage::IsValidOffset(TSeqNo seqNo) const {
    return !(EndSeqNo() < seqNo);
}

IStoragePtr CreateMemStorage(
        const TStorageShardId& shardId,
        const TBytes perShardMemoryLimit,
        const ui32 maxChunks,
        const TOffsetsSettings& offsetsSettings,
        TMemoryUsageInfoPtr totalMemoryUsageInfo,
        IStorageUpdateListener* listener)
{
    auto memoryUsageInfo = totalMemoryUsageInfo ? totalMemoryUsageInfo : new TMemoryUsageInfo(nullptr);
    auto listenerToPass = (listener != nullptr) ? listener : TDummyStorageListener::Get();

    return new TMemStorage(shardId, perShardMemoryLimit, maxChunks, offsetsSettings, memoryUsageInfo, listenerToPass);
}

} // namespace NAgent
} // namespace NSolomon
