#pragma once

#include "common.h"

#include <util/generic/hash.h>
#include <util/generic/yexception.h>
#include <util/stream/str.h>
#include <util/system/spinlock.h>

#include <atomic>

namespace NSolomon::NAgent {

class IMemoryUsageInfoListener {
public:
    virtual ~IMemoryUsageInfoListener() = default;

    virtual void OnSizeChanged(TBytes memorySize) noexcept = 0;
    virtual void OnLimitSet(TBytes memorySize) noexcept = 0;
    virtual void OnBytesWritten(TBytes memorySize) noexcept = 0;
    virtual void OnBytesEvicted(TBytes memorySize) noexcept = 0;
    virtual void OnBytesRead(TBytes memorySize) noexcept = 0;
};

using IMemoryUsageInfoListenerPtr = THolder<IMemoryUsageInfoListener>;

struct TSizeInfo {
    TBytes Total{0};
    TBytes PerShard{0};
};

class TMemoryUsageInfo: public TAtomicRefCount<TMemoryUsageInfo> {
public:
    TMemoryUsageInfo(IMemoryUsageInfoListenerPtr&& listener = nullptr, TBytes totalSizeLimit = MAX_STORAGE_LIMIT)
        : UpdateListener_{std::move(listener)}
        , TotalMemoryUsageLimit_{totalSizeLimit}
    {
        if (UpdateListener_) {
            UpdateListener_->OnLimitSet(totalSizeLimit);
        }
    }

    void BytesWritten(TBytes bytes) {
        if (UpdateListener_) {
            UpdateListener_->OnBytesWritten(bytes);
        }
    }

    void BytesEvicted(TBytes bytes) {
        if (UpdateListener_) {
            UpdateListener_->OnBytesEvicted(bytes);
        }
    }

    void BytesRead(TBytes bytes) {
        if (UpdateListener_) {
            UpdateListener_->OnBytesRead(bytes);
        }
    }

    bool TryIncreaseSize(TBytes bytes) {
        TBytes currentlyUsing = CurrentMemoryUsage_.load(std::memory_order_relaxed);

        while (currentlyUsing <= TotalMemoryUsageLimit_) {
            while (!CurrentMemoryUsage_.compare_exchange_weak(currentlyUsing, currentlyUsing + bytes, std::memory_order_acquire, std::memory_order_acquire)) {
            }

            if (UpdateListener_) {
                UpdateListener_->OnSizeChanged(currentlyUsing + bytes);
            }

            return true;
        }

        return false;
    }

    void DecreaseSize(TBytes bytes) {
        TBytes currentlyUsing = CurrentMemoryUsage_.load(std::memory_order_relaxed);
        Y_ENSURE((currentlyUsing - bytes) >= sizeof(*this), "Cannot subtract more than is held");

        while (!CurrentMemoryUsage_.compare_exchange_weak(currentlyUsing, currentlyUsing - bytes, std::memory_order_acquire, std::memory_order_acquire)) {
        }

        if (UpdateListener_) {
            UpdateListener_->OnSizeChanged(currentlyUsing - bytes);
        }
    }

    TBytes TotalMemoryUsage() {
        return CurrentMemoryUsage_.load(std::memory_order_relaxed);
    }

    TBytes CalculateFreeMemory() {
        return TotalMemoryUsageLimit_ - CurrentMemoryUsage_.load(std::memory_order_relaxed);
    }

private:
    IMemoryUsageInfoListenerPtr UpdateListener_{nullptr};
    TBytes TotalMemoryUsageLimit_{MAX_STORAGE_LIMIT};
    std::atomic<ui64> CurrentMemoryUsage_{sizeof(TMemoryUsageInfo)}; // in bytes
};

using TMemoryUsageInfoPtr = TIntrusivePtr<TMemoryUsageInfo>;

} // namespace NSolomon::NAgent
