#pragma once

#include "sequence_number.h"

#include <solomon/agent/misc/labels_match.h>

#include <util/datetime/base.h>

namespace NSolomon {
namespace NAgent {

class TQuery {
public:
    TQuery() = default;

    explicit TQuery(const TLabels& labels)
        : Labels_(labels)
    {
    }

    explicit TQuery(TLabels&& labels)
        : Labels_(std::move(labels))
    {
    }

    TLabelsMatcher ToMatcher() const {
        return CreateMatcher(MatchType(), Labels());
    }

    // -- setters ------------------------------------------------------------

    TQuery& Labels(const TLabels& labels) {
        Labels_ = labels;
        return *this;
    }

    TQuery& Labels(TLabels&& labels) {
        Labels_ = std::move(labels);
        return *this;
    }

    TQuery& AddLabel(const TLabel& label) {
        Labels_.Add(label);
        return *this;
    }

    TQuery& AddLabel(TLabel&& label) {
        Labels_.Add(std::move(label));
        return *this;
    }

    TQuery& AddLabel(TStringBuf name, TStringBuf value) {
        Labels_.Add(name, value);
        return *this;
    }

    TQuery& Offset(TSeqNo offset) {
        Offset_ = offset;
        return *this;
    }

    TQuery& MatchType(EMatchType matchType) {
        MatchType_ = matchType;
        return *this;
    }

    TQuery& Limit(ui32 limit) {
        Limit_ = limit;
        return *this;
    }

    TQuery& ConsumerId(TMaybe<TString> consumerId) {
        ConsumerId_ = consumerId;
        return *this;
    }

    // -- getters ------------------------------------------------------------

    const TLabels& Labels() const noexcept {
        return Labels_;
    }

    TMaybe<TSeqNo> Offset() const noexcept {
        return Offset_;
    }

    EMatchType MatchType() const noexcept {
        return MatchType_;
    }

    ui32 Limit() const noexcept {
        return Limit_;
    }

    const TMaybe<TString>& ConsumerId() const noexcept {
        return ConsumerId_;
    }

private:
    TLabels Labels_;
    EMatchType MatchType_ = EMatchType::EXACT;
    TMaybe<TSeqNo> Offset_;
    ui32 Limit_ = Max<ui32>();
    TMaybe<TString> ConsumerId_;
};

} // namespace NAgent
} // namespace NSolomon
