#pragma once

#include <util/generic/yexception.h>
#include <util/string/builder.h>
#include <util/system/types.h>

namespace NSolomon {
namespace NAgent {

class TSeqNo {
public:
    TSeqNo() = default;

    explicit constexpr TSeqNo(ui64 value)
        : Value_(value)
    {
    }

    TSeqNo(ui64 chunkOffset, ui64 metricOffset) {
        SetChunkOffset(chunkOffset);
        SetMetricOffset(metricOffset);
    }

    TSeqNo(const TSeqNo&) = default;
    TSeqNo& operator=(const TSeqNo&) = default;

    void SetChunkOffset(ui64 v) {
        Y_ENSURE(v <= MAX_CHUNK_OFFSET, "chunk offset value exceeds limit");
        Value_ = (Value_ & METRIC_OFFSET_MASK) | (v & CHUNK_OFFSET_MASK);
    }

    void SetMetricOffset(ui64 v) {
        Y_ENSURE(v <= MAX_METRIC_OFFSET, "metric offset value exceeds limit");
        Value_ = (Value_ & CHUNK_OFFSET_MASK) | (v << CHUNK_OFFSET_SIZE);
    }

    explicit operator ui64() const {
        return Value_;
    }

    ui64 ChunkOffset() const {
        return Value_ & CHUNK_OFFSET_MASK;
    }

    ui64 MetricOffset() const {
        return Value_ >> CHUNK_OFFSET_SIZE;
    }

    TSeqNo NextChunkNo() const {
        return {ChunkOffset() + 1, 0};
    }

    TSeqNo NextMetricNo() const {
        return {ChunkOffset(), MetricOffset() + 1};
    }

    static constexpr TSeqNo Max() {
        return TSeqNo{std::numeric_limits<decltype(Value_)>::max()};
    }

    static constexpr size_t MaxMetricOffset() {
        return MAX_METRIC_OFFSET;
    }

private:
    ui64 Value_ {0};

    static constexpr auto CHUNK_OFFSET_SIZE = 48;
    static constexpr auto METRIC_OFFSET_SIZE = 16;
    static_assert(CHUNK_OFFSET_SIZE + METRIC_OFFSET_SIZE == 64, "Total size must be 8 bytes");

    static constexpr ui64 CHUNK_OFFSET_MASK = ~(~0ull << CHUNK_OFFSET_SIZE);
    static constexpr ui64 METRIC_OFFSET_MASK = ~CHUNK_OFFSET_MASK;

    static constexpr ui64 MAX_METRIC_OFFSET = ~0ull >> CHUNK_OFFSET_SIZE;
    static constexpr ui64 MAX_CHUNK_OFFSET = ~0ull >> METRIC_OFFSET_SIZE;

};

inline bool operator<(TSeqNo lhs, TSeqNo rhs) {
    return lhs.ChunkOffset() < rhs.ChunkOffset()
        || (lhs.ChunkOffset() == rhs.ChunkOffset() && lhs.MetricOffset() < rhs.MetricOffset());
}

inline bool operator==(TSeqNo lhs, TSeqNo rhs) {
    return ui64(lhs) == ui64(rhs);
}

inline bool operator!=(TSeqNo lhs, TSeqNo rhs) {
    return !(ui64(lhs) == ui64(rhs));
}

inline bool operator<=(TSeqNo lhs, TSeqNo rhs) {
    return (lhs < rhs) || (lhs == rhs);
}

inline bool operator>(TSeqNo lhs, TSeqNo rhs) {
    return !(lhs < rhs || lhs == rhs);
}

} // namespace NAgent
} // namespace NSolomon
