#include "sharded_storage.h"

#include <solomon/agent/protos/storage_config.pb.h>

#include <library/cpp/messagebus/actor/actor.h>

namespace NSolomon {
namespace NAgent {

namespace {

TBytes ParseSizeLimitString(const TString& sizeString) {
    if (sizeString.empty()) {
        return MAX_STORAGE_LIMIT;
    }

    TBytes limit = ParseHumanReadableSize(sizeString);
    if (limit > MAX_STORAGE_LIMIT) {
        SA_LOG(DEBUG) << "size limit is greater than max: " << sizeString << ". Falling back to the max allowed value";
        limit = MAX_STORAGE_LIMIT;
    }

    return limit;
}

} // namespace


TShardedStorage::TShardedStorage(
        const TStorageConfig& config,
        TTimerDispatcherPtr timerDispatcher)
    : Config_{config}
    , Registry_{NMonitoring::TMetricRegistry::Instance()}
    , TimerDispatcher_{timerDispatcher}
{
    bool hasLimitOptions = config.HasLimit() || config.GetShard().size() > 0;
    Y_ENSURE(!(config.GetBufferSize() > 0 && hasLimitOptions),
            "either BufferSize or Limit should be specified, not both");

    if (config.GetBufferSize() > 0) {
        SA_LOG(WARN) << "Storage { BufferSize: ... } field is DEPRECATED! If you have to specify limits, use a new format"
                     << " instead: https://nda.ya.ru/3VmrL2";

        TotalMemoryUsageInfo_ = new TMemoryUsageInfo{CreateMemoryUsageInfoListener(*Registry_)};
        MaxChunks_ = config.GetBufferSize();
    } else {
        MaxChunks_ = Max<ui32>();
        TBytes totalLimit = MAX_STORAGE_LIMIT;

        if (config.HasLimit()) {
            totalLimit = ParseSizeLimitString(Strip(config.GetLimit().GetTotal()));

            if (totalLimit == 0) {
                SA_LOG(DEBUG) << "Total limit is 0, falling back to the max value";
                totalLimit = MAX_STORAGE_LIMIT;
            }

            ShardDefaultLimit_ = ParseSizeLimitString(Strip(config.GetLimit().GetShardDefault()));

            if (ShardDefaultLimit_ == 0) {
                SA_LOG(DEBUG) << "ShardDefault limit is 0, falling back to the max value";
                ShardDefaultLimit_ = MAX_STORAGE_LIMIT;
            }
        }

        TotalMemoryUsageInfo_ = new TMemoryUsageInfo{CreateMemoryUsageInfoListener(*Registry_), totalLimit};

        for (auto shardOptions: config.GetShard()) {
            TString project = Strip(shardOptions.GetProject());
            TString service = Strip(shardOptions.GetService());

            Y_ENSURE(!project.empty(), "Empty Project value in shard options");
            Y_ENSURE(!service.empty(), "Empty Service value in shard options");

            TStorageShardId shardId{ project, service };
            TBytes limit = ParseSizeLimitString(Strip(shardOptions.GetLimit()));

            if (limit == 0) {
                SA_LOG(DEBUG) << "Storage limit for " << shardId << " is 0. Falling back to the max value";
                limit = MAX_STORAGE_LIMIT;
            }

            auto [_, isInserted] = ShardsLimits_.emplace(shardId, limit);
            Y_ENSURE(isInserted,
                    "duplicated ShardLimit for Project=" << shardId.Project << " and Service=" << shardId.Service);
        }

        TString logMsg =
            TStringBuilder() << "creating a sharded storage with limits"
                             << " Total: " << totalLimit << " bytes; ShardDefault: " << ShardDefaultLimit_ << " bytes";
        if (ShardsLimits_.size()) {
            logMsg += "; Shards: [";
            bool first = true;

            for (auto& it: ShardsLimits_) {
                if (!first) {
                    logMsg += ", ";
                }
                logMsg += "{Project=" + it.first.Project;
                logMsg += ", Service=" + it.first.Service;
                logMsg += ", Limit=" + ToString(it.second) + " bytes}";

                first = false;
            }
        }

        SA_LOG(DEBUG) << logMsg;
    }

    if (Config_.HasOffsets()) {
        try {
            const auto &offsetsConfig = Config_.GetOffsets();

            if (const TString& softTTLStr = offsetsConfig.GetSoftTTL()) {
                TDuration softTTL = TDuration::Parse(softTTLStr);
                if (softTTL) {
                    OffsetsSettings_.SoftTTL = softTTL;
                }
            }

            if (const TString& hardTTLStr = offsetsConfig.GetHardTTL()) {
                TDuration hardTTL = TDuration::Parse(hardTTLStr);
                if (hardTTL) {
                    OffsetsSettings_.HardTTL = hardTTL;
                }
            }

            const TDuration hardTTL = OffsetsSettings_.HardTTL;
            const TDuration softTTL = OffsetsSettings_.SoftTTL;

            Y_ENSURE(hardTTL > softTTL,
                    "HardTTL(" << hardTTL.ToString() << ") cannot be less than or equal to"
                    " SoftTTL(" << softTTL.ToString() << ")");

            TDuration watchInterval = TDuration::Parse(offsetsConfig.GetWatchInterval());
            if (watchInterval) {
                OffsetsSettings_.WatchInterval = watchInterval;
            }
        } catch (...) {
            ythrow yexception() << "failed to parse the Offsets section in the config: " << CurrentExceptionMessage();
        }
    }
}

void TShardedStorage::ForEachShard(TShardConsumer&& consumer) const {
    with_lock (ShardsLock_) {
        for (const auto& it: Shards_) {
            const TStorageShardId& shardId = it.first;
            consumer(shardId.Project, shardId.Service);
        }
    }
}

TFindResult TShardedStorage::Find(
        const TString& project, const TString& service,
        const TQuery& query, NMonitoring::IMetricConsumer* c,
        const TFindOptions& options)
{
    TStorageShardId shardId = { project, service };
    auto shardStorage = GetShardStorage(shardId);
    return shardStorage->Find(query, c, options);
}

TReadResult TShardedStorage::Read(
        const TString& project, const TString& service,
        const TQuery& query, NMonitoring::IMetricConsumer* c,
        const TReadOptions& options)
{
    TStorageShardId shardId = { project, service };
    auto shardStorage = GetShardStorage(shardId);
    return shardStorage->Read(query, c, options);
}

void TShardedStorage::Commit(TString project, TString service, const TString& consumerId, TSeqNo seqNo) {
    TStorageShardId shardId {project, service};
    auto* shardStorage = GetShardStorage(shardId);
    shardStorage->Commit(consumerId, seqNo);
}

IStorageMetricsConsumerPtr TShardedStorage::CreateConsumer(TInstant) {
    ythrow yexception() << "CreateConsumer() is not supported. Call CreateConsumer(project, service) instead";
}

IStorageMetricsConsumerPtr TShardedStorage::CreateShardConsumer(
        const TString& project,
        const TString& service,
        TInstant defaultTs)
{
    Y_ENSURE(!project.empty() && !service.empty(), "incomplete shardId");

    auto shardStorage = GetShardStorage({project, service});
    return shardStorage->CreateConsumer(defaultTs);
}

void TShardedStorage::Delete(const TQuery&, const TDeleteOptions&) {
    Y_FAIL("not implemented");
}

void TShardedStorage::Commit(const TString&, TSeqNo) {
    Y_FAIL("not implemented");
}

namespace {
    THolder<NActor::TExecutor> actorsExecutor;

    class TConsumerProvider: public IStorageConsumerProvider {
    public:
        TConsumerProvider(IShardConsumerProvider* shardConsumerProvider, const TString& project, const TString& service)
            : ShardConsumerProvider_{shardConsumerProvider}
            , Project_{project}
            , Service_{service}
        {
        }

        IStorageMetricsConsumerPtr CreateConsumer(TInstant defaultTs) override {
            return ShardConsumerProvider_->CreateShardConsumer(Project_, Service_, defaultTs);
        }

    private:
        IShardConsumerProvider* ShardConsumerProvider_;
        const TString Project_;
        const TString Service_;
    };

} // namespace

IStorage* TShardedStorage::GetShardStorage(const TStorageShardId& shardId) {
    with_lock (ShardsLock_) {
        auto it = Shards_.find(shardId);
        if (it != Shards_.end()) {
            return it->second.Get();
        } else {
            StatusListeners_.emplace_back(CreateStorageShardUpdateListener(shardId, *Registry_));
            std::pair<decltype(Shards_)::iterator, bool> res;

            TBytes shardLimit = ShardDefaultLimit_;
            if (auto shardLimitIt = ShardsLimits_.find(shardId); shardLimitIt != ShardsLimits_.end()) {
                shardLimit = shardLimitIt->second;
            }

            if (Config_.HasAggregationOptions()) {
                if (!actorsExecutor) {
                    actorsExecutor.Reset(new NActor::TExecutor(1));
                }
                res = Shards_.emplace(shardId, CreateAggregatingMemStorage(shardId,
                                                                           shardLimit, MaxChunks_,
                                                                           OffsetsSettings_,
                                                                           TotalMemoryUsageInfo_,
                                                                           TimerDispatcher_,
                                                                           actorsExecutor.Get(),
                                                                           StatusListeners_.back().Get()));
            } else {
                res = Shards_.emplace(shardId, CreateMemStorage(shardId, shardLimit, MaxChunks_, OffsetsSettings_,
                                                                TotalMemoryUsageInfo_, StatusListeners_.back().Get()));
            }
            return res.first->second.Get();
        }
    }
}

IStorageConsumerProviderPtr CreateStorageConsumerProviderForShard(
        IShardConsumerProvider* storage,
        const TString& project,
        const TString& service)
{
    return MakeHolder<TConsumerProvider>(storage, project, service);
}

} // namespace NAgent
} // namespace NSolomon
