#pragma once

#include <solomon/agent/lib/storage/common.h>
#include <solomon/agent/lib/storage/memory_usage.h>
#include <solomon/agent/lib/storage/storage.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/timer_dispatcher.h>
#include <solomon/agent/protos/storage_config.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <util/string/cast.h>
#include <util/string/strip.h>
#include <util/system/spinlock.h>

namespace NSolomon {
namespace NAgent {

IMemoryUsageInfoListenerPtr CreateMemoryUsageInfoListener(NMonitoring::TMetricRegistry& metricRegistry);
IStorageUpdateListenerPtr CreateStorageShardUpdateListener(
        const TStorageShardId& shardId,
        NMonitoring::TMetricRegistry& registry);

class IShardConsumerProvider {
public:
    virtual IStorageMetricsConsumerPtr CreateShardConsumer(
            const TString& project,
            const TString& service,
            TInstant defaultTs = TInstant::Zero()) = 0;
};

class TShardedStorage: public IStorageWriter, public IShardConsumerProvider {
    using TShardConsumer = std::function<void(const TString&, const TString&)>;

public:
    explicit TShardedStorage(
            const TStorageConfig& config,
            TTimerDispatcherPtr timerDispatcher = nullptr);

    void ForEachShard(TShardConsumer&& consumer) const;

    // pseudo-IStorageReader interface
    TFindResult Find(
        const TString& project,
        const TString& service,
        const TQuery& query,
        NMonitoring::IMetricConsumer* c,
        const TFindOptions& options = TFindOptions());

    TReadResult Read(
        const TString& project,
        const TString& service,
        const TQuery& query,
        NMonitoring::IMetricConsumer* c,
        const TReadOptions& options = TReadOptions());

    void Commit(TString project, TString service, const TString& consumerId, TSeqNo seqNo);

private:
    IStorageMetricsConsumerPtr CreateConsumer(TInstant defaultTs) override;
    IStorageMetricsConsumerPtr CreateShardConsumer(
            const TString& project,
            const TString& service,
            TInstant defaultTs) override;

    void Delete(
            const TQuery& query,
            const TDeleteOptions& options = TDeleteOptions()) override;

    void Commit(const TString& consumerId, TSeqNo seqNo) override;

    IStorage* GetShardStorage(const TStorageShardId& shardId);

private:
    class TShardedStorageMetricsConsumer;

    const TStorageConfig& Config_;
    TOffsetsSettings OffsetsSettings_;
    TMemoryUsageInfoPtr TotalMemoryUsageInfo_;
    ui32 MaxChunks_;
    TBytes ShardDefaultLimit_{MAX_STORAGE_LIMIT};
    THashMap<TStorageShardId, TBytes> ShardsLimits_;

    NMonitoring::TMetricRegistry* Registry_;
    TTimerDispatcherPtr TimerDispatcher_;
    TVector<IStorageUpdateListenerPtr> StatusListeners_;

    THashMap<TStorageShardId, IStoragePtr> Shards_;
    TAdaptiveLock ShardsLock_;
};

IStorageConsumerProviderPtr CreateStorageConsumerProviderForShard(
        IShardConsumerProvider* storage,
        const TString& project,
        const TString& service);

} // namespace NAgent
} // namespace NSolomon
