#pragma once

#include "common.h"
#include "memory_usage.h"
#include "query.h"
#include "sequence_number.h"

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/monlib/metrics/metric_consumer.h>

#include <util/generic/ptr.h>
#include <util/datetime/base.h>

namespace NProto {
    class TMultiSample;
    class TMultiSamplesList;
} // namespace NProto

namespace NActor {
    class TExecutor;
} // namespace NActor

namespace NSolomon {
namespace NAgent {

class IStorageUpdateListener {
public:
    virtual ~IStorageUpdateListener() = default;

    virtual void OnFetcherUpdated(const TString& fetcherId, ui64 chunkOffset) noexcept = 0;

    virtual void OnPointsAdded(ui64 count) noexcept = 0;
    virtual void OnPointsRemoved(ui64 count) noexcept = 0;

    virtual void OnSizeChanged(TBytes Size) noexcept = 0;
    virtual void OnBytesWritten(TBytes memorySize) noexcept = 0;
    virtual void OnBytesEvicted(TBytes memorySize) noexcept = 0;
    virtual void OnBytesRead(TBytes memorySize) noexcept = 0;

    virtual void OnAdded(ui64 chunkCount) noexcept = 0;

    // number of chunks evicted because of lack of storage space in the buffer
    virtual void OnOverflow(ui64 chunkCount) noexcept = 0;
    virtual void OnRemovedReadChunks(ui64 chunkCount) noexcept = 0;

    virtual void OnLimitSet(TBytes memorySize) noexcept = 0;
};

using IStorageUpdateListenerPtr = THolder<IStorageUpdateListener>;

///////////////////////////////////////////////////////////////////////////////
// options
///////////////////////////////////////////////////////////////////////////////
struct TReadOptions {
};

struct TWriteOptions {
};

struct TFindOptions {
};

struct TDeleteOptions {
};

///////////////////////////////////////////////////////////////////////////////
// operation result types
///////////////////////////////////////////////////////////////////////////////
struct TReadResult {
    TSeqNo SeqNo;
    bool HasMore;
    ui32 NumOfMetrics;
};

struct TFindResult {
    TSeqNo SeqNo;
    bool HasMore;
};

///////////////////////////////////////////////////////////////////////////////
// IStorageReader
///////////////////////////////////////////////////////////////////////////////
class IStorageReader {
public:
    virtual ~IStorageReader() = default;

    /// @returns Index of the first unread metric.
    virtual TReadResult Read(
            const TQuery& query,
            NMonitoring::IMetricConsumer* c,
            const TReadOptions& options = TReadOptions()) = 0;

    virtual TFindResult Find(
            const TQuery& query,
            NMonitoring::IMetricConsumer* c,
            const TFindOptions& options = TFindOptions()) = 0;
};


///////////////////////////////////////////////////////////////////////////////
// IStorageWriter
///////////////////////////////////////////////////////////////////////////////

class IStorageMetricsConsumer: public NMonitoring::IMetricConsumer {
public:
    /**
     * Flush data to the associated storage. To rollback, simply destruct a consumer object
     */
    virtual void Flush() = 0;
};

using IStorageMetricsConsumerPtr = THolder<IStorageMetricsConsumer>;

class IStorageConsumerProvider {
public:
    virtual ~IStorageConsumerProvider() = default;

    virtual IStorageMetricsConsumerPtr CreateConsumer(TInstant defaultTs = TInstant::Zero()) = 0;
};

using IStorageConsumerProviderPtr = THolder<IStorageConsumerProvider>;

class IStorageWriter: public IStorageConsumerProvider {
public:
    virtual ~IStorageWriter() = default;

    virtual void Delete(
            const TQuery& query,
            const TDeleteOptions& options = TDeleteOptions()) = 0;

    virtual void Commit(const TString& consumerId, TSeqNo seqNo) = 0;
};


///////////////////////////////////////////////////////////////////////////////
// IStorage
///////////////////////////////////////////////////////////////////////////////
class IStorage: public IStorageReader, public IStorageWriter, public TAtomicRefCount<IStorage> {
};

using IStoragePtr = TIntrusivePtr<IStorage>;

IStoragePtr CreateMemStorage(
        const TStorageShardId& shardId,
        const TBytes perShardMemoryLimit,
        const ui32 maxChunks,
        const TOffsetsSettings& offsetsSettings = {},
        TMemoryUsageInfoPtr totalMemoryUsageInfo = new TMemoryUsageInfo{},
        IStorageUpdateListener* statusListener = nullptr);
IStoragePtr CreateAggregatingMemStorage(
        const TStorageShardId& shardId,
        const TBytes perShardMemoryLimit,
        const ui32 maxChunks,
        const TOffsetsSettings& offsetsSettings = {},
        TMemoryUsageInfoPtr totalMemoryUsageInfo = new TMemoryUsageInfo{},
        TIntrusivePtr<class TTimerDispatcher> timerDispatcher = nullptr,
        NActor::TExecutor* actorsExecutor = nullptr,
        IStorageUpdateListener* statusListener = nullptr);

} // namespace NAgent
} // namespace NSolomon
