#pragma once

#include <solomon/agent/lib/storage/sharded_storage.h>
#include <solomon/agent/misc/labels.h>

#include <library/cpp/monlib/metrics/metric.h>

#include <util/datetime/base.h>
#include <util/generic/vector.h>

namespace NSolomon::NAgent::NTest {

///////////////////////////////////////////////////////////////////////////////
// TStubStorage
///////////////////////////////////////////////////////////////////////////////
struct TPoint {
    TInstant Time;
    ui64 Value;
};

struct TMetric {
    NMonitoring::EMetricType Type;
    TAgentLabels Labels;
    TVector<TPoint> Points;
};

using TChunk = TVector<TMetric>;
using TChunks = TVector<TChunk>;

class TStubStorage: public IStorageWriter, public IShardConsumerProvider {
public:
    const TChunks Chunks() const noexcept {
        with_lock (Lock_) {
            return Chunks_;
        }
    }

    class TStubStorageConsumer: public IStorageMetricsConsumer {
    public:
        TStubStorageConsumer(TStubStorage* storage, TInstant defaultTs)
            : Storage_{storage}
            , DefaultTs_{defaultTs}
        {}

        void OnStreamBegin() override {}
        void OnStreamEnd() override {}

        void OnCommonTime(TInstant time) override {
            CommonTime_ = time;
        }

        void OnMetricBegin(NMonitoring::EMetricType type) override {
            State_ = METRIC;
            Current_ = &Metrics_.emplace_back();
            Current_->Type = type;
        }

        void OnMetricEnd() override {
            State_ = ROOT;
        }

        void OnLabelsBegin() override {
            if (State_ == ROOT) {
                State_ = COMMON_LABELS;
            }
        }

        void OnLabelsEnd() override {
            if (State_ == METRIC) {
                for (auto&& label: CommonLabels_) {
                    Current_->Labels.Add(label);
                }
            } else if (State_ == COMMON_LABELS) {
                State_ = ROOT;
            }
        }

        void OnLabel(TStringBuf name, TStringBuf value) override {
            if (State_ == COMMON_LABELS) {
                CommonLabels_.Add(name, value);
            } else {
                Current_->Labels.Add(name, value);
            }
        }

        void OnDouble(TInstant, double) override {
            ythrow yexception() << "not supported";
        }
        void OnInt64(TInstant, i64) override {
            ythrow yexception() << "not supported";
        }
        void OnHistogram(TInstant, NMonitoring::IHistogramSnapshotPtr) override {
            ythrow yexception() << "not supported";
        }

        void OnSummaryDouble(TInstant, NMonitoring::ISummaryDoubleSnapshotPtr) override {
            ythrow yexception() << "not supported";
        }

        void OnLogHistogram(TInstant, NMonitoring::TLogHistogramSnapshotPtr) override {
            ythrow yexception() << "not supported";
        }

        void OnUint64(TInstant time, ui64 value) override {
            time = ComputeTime(NMonitoring::EMetricType::COUNTER, time, CommonTime_, DefaultTs_);
            Current_->Points.emplace_back(TPoint{time, value});
        }

        void Flush() override {
            Storage_->Write(std::move(Metrics_));
        }

    public:
        TChunk Metrics_;

    private:
        enum EState {
            ROOT,
            COMMON_LABELS,
            METRIC
        };

        EState State_;
        TLabels CommonLabels_;
        TInstant CommonTime_;

        TStubStorage* Storage_;
        TInstant DefaultTs_;
        TMetric* Current_;
    };

    IStorageMetricsConsumerPtr CreateConsumer(TInstant) override {
        ythrow yexception() << "not supported. Call CreateShardConsumer(project, service) instead";
    }

    IStorageMetricsConsumerPtr CreateShardConsumer(const TString&, const TString&,
                                                   TInstant defaultTs = TInstant::Zero()) override {
        return MakeHolder<TStubStorageConsumer>(this, defaultTs);
    }

    void Write(TChunk&& chunk) {
        with_lock (Lock_) {
            Chunks_.emplace_back(std::move(chunk));
        }
    }

    void Delete(const TQuery&, const TDeleteOptions&) override {
        Y_FAIL("not implemented");
    }

    void Commit(const TString&, TSeqNo) override {
        Y_FAIL("not implemented");
    }

private:
    TAdaptiveLock Lock_;
    TChunks Chunks_;
};

}
