#include "transforming_storage.h"

#include <library/cpp/monlib/encode/protobuf/protobuf.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon;
using namespace NAgent;

using NMonitoring::NProto::TMultiSamplesList;
using NMonitoring::NProto::TPoint;

TVector<double> TransformToDoubles(TMultiSamplesList& samples) {
    TVector<double> points;
    for (auto i = 0u; i < samples.SamplesSize(); ++i) {
        auto& ps = samples.GetSamples()[i].GetPoints();
        Transform(std::begin(ps), std::end(ps), std::back_inserter(points),
                  [] (const auto& point) {
                      return point.GetFloat64();
                  });
    }
    return points;
}

struct TMemory {
    IStoragePtr operator()(ui32 bufferSize = 16) const {
        return CreateMemStorage({"project", "service"}, MAX_STORAGE_LIMIT, bufferSize);
    }
};

struct TTransforming {
    IStoragePtr operator()(ui32 bufferSize = 16) const {
        return new TTransformingStorage(bufferSize);
    }
};

using TStorageTypes = ::testing::Types<TMemory, TTransforming>;
TYPED_TEST_SUITE(TStorageTest, TStorageTypes);

template <typename TStorageFactory>
class TStorageTest: public ::testing::Test {
protected:
    void WriteTwoMetrics(IStorageMetricsConsumer* consumer, TInstant ts, double value1, uint64_t value2) {
        consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);

        consumer->OnLabelsBegin();
        consumer->OnLabel("my", "gauge");
        consumer->OnLabel("label", "value");
        consumer->OnLabelsEnd();

        consumer->OnDouble(ts, value1);

        consumer->OnMetricEnd();

        // -----------

        consumer->OnMetricBegin(NMonitoring::EMetricType::COUNTER);

        consumer->OnLabelsBegin();
        consumer->OnLabel("my", "counter");
        consumer->OnLabel("label", "value");
        consumer->OnLabelsEnd();

        consumer->OnUint64(ts, value2);

        consumer->OnMetricEnd();
    }

    void WriteData(IStorage& storage, TInstant ts, TInstant nextTs) {
        IStorageMetricsConsumerPtr consumer = storage.CreateConsumer();
        consumer->OnStreamBegin();

        WriteTwoMetrics(consumer.Get(), ts, 12.34, 1234);
        WriteTwoMetrics(consumer.Get(), nextTs, 12.35, 1235);

        consumer->OnStreamEnd();
        consumer->Flush();
    }

    void WriteDataWithNoTsButDefault(IStorage& storage, TInstant defaultTs) {
        IStorageMetricsConsumerPtr consumer = storage.CreateConsumer(defaultTs);
        consumer->OnStreamBegin();

        WriteTwoMetrics(consumer.Get(), defaultTs, 12.34, 1234);

        consumer->OnStreamEnd();
        consumer->Flush();
    }

    void WriteSamplesSimple(IStorage& storage, ui32 count) {
        for (auto i = 0u; i < count; ++i) {
            IStorageMetricsConsumerPtr consumer = storage.CreateConsumer();
            consumer->OnStreamBegin();
            consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);

            consumer->OnLabelsBegin();
            consumer->OnLabel("label", "value");
            consumer->OnLabelsEnd();

            consumer->OnDouble(TInstant::MilliSeconds(i), i);

            consumer->OnMetricEnd();
            consumer->OnStreamEnd();
            consumer->Flush();
        }
    }

    void WriteWithDifferentLabels(IStorage& storage, ui32 count) {
        for (auto i = 0u; i < count; ++i) {
            IStorageMetricsConsumerPtr consumer = storage.CreateConsumer();
            consumer->OnStreamBegin();
            consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);

            consumer->OnLabelsBegin();
            consumer->OnLabel("my", "labels");
            consumer->OnLabel("label", TString{"value_"} + ToString(i));
            consumer->OnLabelsEnd();

            consumer->OnDouble(TInstant::MilliSeconds(i), i);

            consumer->OnMetricEnd();
            consumer->OnStreamEnd();
            consumer->Flush();
        }
    }

protected:
    TStorageFactory StorageFactory_;
};

TYPED_TEST(TStorageTest, ReadMetricsTest) {
    IStoragePtr storage = this->StorageFactory_();
    TInstant now = TInstant::Now();
    TInstant nextNow = now + TDuration::Seconds(1);

    this->WriteData(*storage, now, nextNow);

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS);
        storage->Read(query, encoder.Get());

        TVector<TPoint> gaugePoints;
        TVector<TPoint> counterPoints;

        for (const auto& sample: samples.GetSamples()) {
            auto type = sample.GetMetricType();
            if (type == NMonitoring::NProto::GAUGE) {
                const auto& labels = sample.GetLabels();
                ASSERT_EQ(labels.size(), 2);

                const auto& l1 = labels.Get(0);
                ASSERT_EQ(l1.GetName(), "label");
                ASSERT_EQ(l1.GetValue(), "value");

                const auto& l2 = labels.Get(1);
                ASSERT_EQ(l2.GetName(), "my");
                ASSERT_EQ(l2.GetValue(), "gauge");

                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(gaugePoints));
            } else if (type == NMonitoring::NProto::COUNTER) {
                const auto& labels = sample.GetLabels();
                ASSERT_EQ(labels.size(), 2);

                const auto& l1 = labels.Get(0);
                ASSERT_EQ(l1.GetName(), "label");
                ASSERT_EQ(l1.GetValue(), "value");

                const auto& l2 = labels.Get(1);
                ASSERT_EQ(l2.GetName(), "my");
                ASSERT_EQ(l2.GetValue(), "counter");

                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(counterPoints));

            } else {
                FAIL() << "unknown type: " << NMonitoring::NProto::EMetricType_Name(type);
            }
        }

        {
            ASSERT_EQ(gaugePoints.size(), 2u);
            const auto p1 = gaugePoints[0];
            ASSERT_EQ(p1.GetTime(), now.MilliSeconds());
            ASSERT_DOUBLE_EQ(p1.GetFloat64(), 12.34);

            const auto p2 = gaugePoints[1];
            ASSERT_EQ(p2.GetTime(), nextNow.MilliSeconds());
            ASSERT_DOUBLE_EQ(p2.GetFloat64(), 12.35);
        }

        {
            ASSERT_EQ(counterPoints.size(), 2u);
            const auto p1 = counterPoints[0];
            ASSERT_EQ(p1.GetTime(), now.MilliSeconds());
            ASSERT_EQ(p1.GetUint64(), 1234u);

            const auto p2 = counterPoints[1];
            ASSERT_EQ(p2.GetTime(), nextNow.MilliSeconds());
            ASSERT_EQ(p2.GetUint64(), 1235u);
        }
    }
}

TYPED_TEST(TStorageTest, ReadMetricsWithOffsetsTest) {
    IStoragePtr storage = this->StorageFactory_();
    TInstant ts11 = TInstant::Now();
    TInstant ts12 = ts11 + TDuration::Seconds(1);

    TInstant ts21 = ts11 + TDuration::Seconds(10);
    TInstant ts22 = ts12 + TDuration::Seconds(10);

    TInstant ts31 = ts21 + TDuration::Seconds(10);
    TInstant ts32 = ts22 + TDuration::Seconds(10);

    this->WriteData(*storage, ts11, ts12);
    this->WriteData(*storage, ts21, ts22);
    this->WriteData(*storage, ts31, ts32);

    storage->Commit("foo", TSeqNo{0});
    storage->Commit("bar", TSeqNo{0});

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);

        TQuery query(TLabels { {"label", "value"} });

        TSeqNo offset = TSeqNo{2, 0};
        query.ConsumerId("bar").Offset(offset);
        query.MatchType(EMatchType::CONTAINS);

        storage->Commit("bar", offset);
        storage->Read(query, encoder.Get());

        TVector<TPoint> gaugePoints;

        for (const auto& sample: samples.GetSamples()) {
            auto type = sample.GetMetricType();
            if (type == NMonitoring::NProto::GAUGE) {
                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(gaugePoints));
            }
        }

        {
            ASSERT_EQ(gaugePoints.size(), 4u);
            const auto p1 = gaugePoints[0];
            ASSERT_EQ(p1.GetTime(), ts21.MilliSeconds());
            ASSERT_DOUBLE_EQ(p1.GetFloat64(), 12.34);

            const auto p2 = gaugePoints[1];
            ASSERT_EQ(p2.GetTime(), ts22.MilliSeconds());
            ASSERT_DOUBLE_EQ(p2.GetFloat64(), 12.35);

            const auto p3 = gaugePoints[2];
            ASSERT_EQ(p3.GetTime(), ts31.MilliSeconds());
            ASSERT_DOUBLE_EQ(p3.GetFloat64(), 12.34);

            const auto p4 = gaugePoints[3];
            ASSERT_EQ(p4.GetTime(), ts32.MilliSeconds());
            ASSERT_DOUBLE_EQ(p4.GetFloat64(), 12.35);
        }
    }

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        TQuery query(TLabels { {"label", "value"} });

        // Chunk offset is less than before (but not zero -- otherwise Agent will think that we've restarted)
        TSeqNo offset = TSeqNo{1, 0};
        query.ConsumerId("bar").Offset(offset);
        query.MatchType(EMatchType::CONTAINS);

        storage->Commit("bar", offset);
        storage->Read(query, encoder.Get());

        TVector<TPoint> gaugePoints;

        for (const auto& sample: samples.GetSamples()) {
            auto type = sample.GetMetricType();
            if (type == NMonitoring::NProto::GAUGE) {
                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(gaugePoints));
            }
        }

        // But still the same data should be given
        {
            ASSERT_EQ(gaugePoints.size(), 4u);
            const auto p1 = gaugePoints[0];
            ASSERT_EQ(p1.GetTime(), ts21.MilliSeconds());
            ASSERT_DOUBLE_EQ(p1.GetFloat64(), 12.34);

            const auto p2 = gaugePoints[1];
            ASSERT_EQ(p2.GetTime(), ts22.MilliSeconds());
            ASSERT_DOUBLE_EQ(p2.GetFloat64(), 12.35);

            const auto p3 = gaugePoints[2];
            ASSERT_EQ(p3.GetTime(), ts31.MilliSeconds());
            ASSERT_DOUBLE_EQ(p3.GetFloat64(), 12.34);

            const auto p4 = gaugePoints[3];
            ASSERT_EQ(p4.GetTime(), ts32.MilliSeconds());
            ASSERT_DOUBLE_EQ(p4.GetFloat64(), 12.35);
        }
    }
}

TYPED_TEST(TStorageTest, ReadMetricsNewConsumerWithNonZeroOffsetTest) {
    IStoragePtr storage = this->StorageFactory_();
    TInstant ts11 = TInstant::Now();
    TInstant ts12 = ts11 + TDuration::Seconds(1);

    TInstant ts21 = ts11 + TDuration::Seconds(10);
    TInstant ts22 = ts12 + TDuration::Seconds(10);

    TInstant ts31 = ts21 + TDuration::Seconds(10);
    TInstant ts32 = ts22 + TDuration::Seconds(10);

    this->WriteData(*storage, ts11, ts12); // TSeqNo: 0
    this->WriteData(*storage, ts21, ts22); // TSeqNo: 1
    this->WriteData(*storage, ts31, ts32); // TSeqNo: 2

    // New consumer, but with non-zero offset
    TString consumerId = "foo";
    TSeqNo offset = TSeqNo{1, 0};

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        TQuery query(TLabels { {"label", "value"} });

        query.ConsumerId(consumerId).Offset(offset);
        query.MatchType(EMatchType::CONTAINS);

        storage->Commit(consumerId, offset);
        storage->Read(query, encoder.Get());

        TVector<TPoint> gaugePoints;
        for (const auto& sample: samples.GetSamples()) {
            auto type = sample.GetMetricType();
            if (type == NMonitoring::NProto::GAUGE) {
                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(gaugePoints));
            }
        }

        // Consumer is new, so regardless of an offset value, all data should be extracted
        {
            ASSERT_EQ(gaugePoints.size(), 6u);
            // TSeqNo: 0
            const auto p1 = gaugePoints[0];
            ASSERT_EQ(p1.GetTime(), ts11.MilliSeconds());
            ASSERT_DOUBLE_EQ(p1.GetFloat64(), 12.34);

            const auto p2 = gaugePoints[1];
            ASSERT_EQ(p2.GetTime(), ts12.MilliSeconds());
            ASSERT_DOUBLE_EQ(p2.GetFloat64(), 12.35);

            // TSeqNo: 1
            const auto p3 = gaugePoints[2];
            ASSERT_EQ(p3.GetTime(), ts21.MilliSeconds());
            ASSERT_DOUBLE_EQ(p3.GetFloat64(), 12.34);

            const auto p4 = gaugePoints[3];
            ASSERT_EQ(p4.GetTime(), ts22.MilliSeconds());
            ASSERT_DOUBLE_EQ(p4.GetFloat64(), 12.35);

            // TSeqNo: 2
            const auto p5 = gaugePoints[4];
            ASSERT_EQ(p5.GetTime(), ts31.MilliSeconds());
            ASSERT_DOUBLE_EQ(p5.GetFloat64(), 12.34);

            const auto p6 = gaugePoints[5];
            ASSERT_EQ(p6.GetTime(), ts32.MilliSeconds());
            ASSERT_DOUBLE_EQ(p6.GetFloat64(), 12.35);
        }
    }
}

TYPED_TEST(TStorageTest, FindMetricsTest) {
    IStoragePtr storage = this->StorageFactory_();
    TInstant now = TInstant::Now();
    TInstant nextNow = now + TDuration::Seconds(1);

    this->WriteData(*storage, now, nextNow);

    TMultiSamplesList samples;
    auto encoder = NMonitoring::EncoderProtobuf(&samples);
    TQuery query(TLabels { {"label", "value"} });
    query.MatchType(EMatchType::CONTAINS);
    storage->Find(query, encoder.Get());

    ASSERT_EQ(samples.SamplesSize(), 2u);
    for (const auto& sample: samples.GetSamples()) {
        auto type = sample.GetMetricType();
        if (type == NMonitoring::NProto::GAUGE) {
            const auto& labels = sample.GetLabels();
            ASSERT_EQ(labels.size(), 2);

            const auto& l1 = labels.Get(0);
            ASSERT_EQ(l1.GetName(), "label");
            ASSERT_EQ(l1.GetValue(), "value");

            const auto& l2 = labels.Get(1);
            ASSERT_EQ(l2.GetName(), "my");
            ASSERT_EQ(l2.GetValue(), "gauge");
        } else if (type == NMonitoring::NProto::COUNTER) {
            const auto& labels = sample.GetLabels();
            ASSERT_EQ(labels.size(), 2);

            const auto& l1 = labels.Get(0);
            ASSERT_EQ(l1.GetName(), "label");
            ASSERT_EQ(l1.GetValue(), "value");

            const auto& l2 = labels.Get(1);
            ASSERT_EQ(l2.GetName(), "my");
            ASSERT_EQ(l2.GetValue(), "counter");
        }
    }
}

TYPED_TEST(TStorageTest, OldRecordsGetEvicted) {
    IStoragePtr storage = this->StorageFactory_(2);
    this->WriteSamplesSimple(*storage, 10);

    TMultiSamplesList samples;
    auto encoder = NMonitoring::EncoderProtobuf(&samples);
    TQuery query(TLabels { {"label", "value"} });
    query.MatchType(EMatchType::CONTAINS)
            .ConsumerId({"test"});
    storage->Read(query, encoder.Get());

    TVector<double> points = TransformToDoubles(samples);
    ASSERT_EQ(points.size(), 2u);
    ASSERT_DOUBLE_EQ(points[0], 8.);
    ASSERT_DOUBLE_EQ(points[1], 9.);
}

// TODO: fix this test, because earlier it was disabled
TYPED_TEST(TStorageTest, ReadRecordsGetEvicted) {
    IStoragePtr storage = this->StorageFactory_(22);
    this->WriteSamplesSimple(*storage, 2);

    TSeqNo nextSeqNo;
    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS)
                .ConsumerId({"test"});

        auto res = storage->Read(query, encoder.Get());
        nextSeqNo = res.SeqNo;

        TVector<double> points = TransformToDoubles(samples);
        ASSERT_EQ(points.size(), 2u);
        ASSERT_DOUBLE_EQ(points[0], 0.);
        ASSERT_DOUBLE_EQ(points[1], 1.);
    }

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);

        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS)
                .ConsumerId({"test"})
                .Offset(nextSeqNo);

        storage->Commit(*query.ConsumerId(), nextSeqNo);
        storage->Read(query, encoder.Get());

        TVector<double> points = TransformToDoubles(samples);
        ASSERT_EQ(points.size(), 0u);
    }

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);

        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS)
                .ConsumerId({""}) // a hack: an invisible consumer who always gets all raw data (even already read)
                .Offset(nextSeqNo);
        storage->Read(query, encoder.Get());

        TVector<double> points = TransformToDoubles(samples);
        // All chunks should be deleted after the last read
        ASSERT_EQ(points.size(), 0u);
    }

    ui64 numOfChunks = 2;
    this->WriteSamplesSimple(*storage, numOfChunks);
    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);

        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS)
                .ConsumerId({"test"})
                .Offset(nextSeqNo);

        storage->Commit(*query.ConsumerId(), nextSeqNo);
        auto res = storage->Read(query, encoder.Get());

        TVector<double> points = TransformToDoubles(samples);
        ASSERT_EQ(points.size(), 2u);
        ASSERT_DOUBLE_EQ(points[0], 0.);
        ASSERT_DOUBLE_EQ(points[1], 1.);
        // Even after the storage was empty, the SeqNo value still rises
        ASSERT_EQ(res.SeqNo.ChunkOffset(), nextSeqNo.ChunkOffset() + numOfChunks);
    }
}

TYPED_TEST(TStorageTest, RangeOperationsTest) {
    constexpr auto SIZE = 10;
    IStoragePtr storage = this->StorageFactory_(SIZE);
    this->WriteWithDifferentLabels(*storage, SIZE);

    TString consumerId = "test";

    TQuery query(TLabels { {"my", "labels"} });
    query.MatchType(EMatchType::CONTAINS)
            .Limit(1)
            .ConsumerId({consumerId});

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        storage->Find(query, encoder.Get());

        ASSERT_EQ(samples.SamplesSize(), 1u);
        const auto& sample = samples.GetSamples()[0];
        ASSERT_EQ(sample.LabelsSize(), 2u);
    }

    {
        storage->Commit(consumerId, TSeqNo{0});
        for (auto i = 0; i < SIZE; ++i) {
            TMultiSamplesList samples;
            auto encoder = NMonitoring::EncoderProtobuf(&samples);
            auto result = storage->Read(query, encoder.Get());

            ASSERT_EQ(samples.SamplesSize(), 1u);
            const auto& sample = samples.GetSamples()[0];
            ASSERT_EQ(sample.PointsSize(), 1u);

            if (i < SIZE - 1) {
                ASSERT_TRUE(result.HasMore);
            } else {
                ASSERT_FALSE(result.HasMore);
            }

            storage->Commit(consumerId, result.SeqNo);
            query.Offset(result.SeqNo).Limit(1);
        }
    }
}

TYPED_TEST(TStorageTest, WriteManyMetricsTest) {
    // expecting 3 chunks here
    constexpr auto SIZE = static_cast<size_t>(Max<ui16>()) * 2 + 1;
    IStoragePtr storage = this->StorageFactory_(SIZE);

    IStorageMetricsConsumerPtr consumer = storage->CreateConsumer();
    consumer->OnStreamBegin();

    for (auto i = 0u; i < SIZE; ++i) {
        consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);

        consumer->OnLabel("label", TStringBuilder() << "value" << i);
        consumer->OnDouble(TInstant::MilliSeconds(i), i);

        consumer->OnMetricEnd();
    }

    consumer->OnStreamEnd();
    consumer->Flush();

    TQuery query;
    TMultiSamplesList samplesToRead;
    {
        auto encoder = NMonitoring::EncoderProtobuf(&samplesToRead);
        storage->Read(query, encoder.Get());
    }

    ASSERT_EQ(samplesToRead.SamplesSize(), SIZE);
}

TYPED_TEST(TStorageTest, WriteNoMetricsTest) {
    IStoragePtr storage = this->StorageFactory_();

    storage->CreateConsumer()->Flush();

    TMultiSamplesList samplesToRead;
    auto encoder = NMonitoring::EncoderProtobuf(&samplesToRead);
    TQuery query(TLabels { {"label", "value"} });
    query.MatchType(EMatchType::CONTAINS);
    storage->Read(query, encoder.Get());

    ASSERT_EQ(samplesToRead.SamplesSize(), 0u);
}

TYPED_TEST(TStorageTest, WriteMetricsWithNoTsButDefaultTest) {
    IStoragePtr storage = this->StorageFactory_();
    TInstant defaultTs = TInstant::Seconds(666);
    this->WriteDataWithNoTsButDefault(*storage, defaultTs);

    {
        TMultiSamplesList samples;
        auto encoder = NMonitoring::EncoderProtobuf(&samples);
        TQuery query(TLabels { {"label", "value"} });
        query.MatchType(EMatchType::CONTAINS);
        storage->Read(query, encoder.Get());

        TVector<TPoint> gaugePoints;
        TVector<TPoint> counterPoints;

        for (const auto& sample: samples.GetSamples()) {
            auto type = sample.GetMetricType();
            if (type == NMonitoring::NProto::GAUGE) {
                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(gaugePoints));
            } else if (type == NMonitoring::NProto::COUNTER) {
                const auto& points = sample.GetPoints();
                Copy(std::begin(points), std::end(points), std::back_inserter(counterPoints));
            } else {
                FAIL() << "unknown type: " << NMonitoring::NProto::EMetricType_Name(type);
            }
        }

        {
            ASSERT_EQ(gaugePoints.size(), 1u);
            const auto p1 = gaugePoints[0];
            ASSERT_EQ(p1.GetTime(), defaultTs.MilliSeconds());
            ASSERT_DOUBLE_EQ(p1.GetFloat64(), 12.34);
        }

        {
            ASSERT_EQ(counterPoints.size(), 1u);
            const auto p1 = counterPoints[0];
            ASSERT_EQ(p1.GetTime(), defaultTs.MilliSeconds());
            ASSERT_EQ(p1.GetUint64(), 1234u);
        }
    }
}
