#include "pool_provider.h"
#include "thread_pool.h"

#include <solomon/agent/misc/logger.h>

#include <util/generic/yexception.h>

namespace NSolomon::NAgent {

class TThreadPoolProviderKeyException: public yexception {};

namespace {

class TLazyThreadPoolProvider: public IThreadPoolProvider {
public:
    explicit TLazyThreadPoolProvider(
            const TThreadPoolProviderConfig& config,
            TThreadPoolStatusListenerFactory listenerFactory = nullptr)
        : StatusListenerFactory_{std::move(listenerFactory)}
    {
        for (const TThreadPoolConfig& poolConfig: config.GetThreadPools()) {
            ThreadPoolConfigs_[poolConfig.GetName()] = poolConfig;
        }
    }

    explicit TLazyThreadPoolProvider(TThreadPoolStatusListenerFactory listenerFactory = nullptr)
        : StatusListenerFactory_(std::move(listenerFactory))
    {
        ThreadPools_[DefaultPoolMeta::Name] = CreateDefaultThreadPool();
    }

    TSimpleSharedPtr<IThreadPool> GetThreadPool(TStringBuf poolName) override {
        if (poolName.empty()) {
            ythrow TThreadPoolProviderKeyException() << "ThreadPoolProvider key cannot be empty";
        }

        with_lock (Lock_) {
            auto poolPtr = ThreadPools_.find(poolName);
            if (poolPtr != ThreadPools_.end()) {
                return poolPtr->second;
            }

            TSimpleSharedPtr<IThreadPool> pool;
            auto poolConfigPtr = ThreadPoolConfigs_.find(poolName);

            if (poolConfigPtr != ThreadPoolConfigs_.end()) {
                pool = CreateThreadPool(poolConfigPtr->second);
            } else if (DefaultPoolMeta::Name == poolName) {
                pool = CreateDefaultThreadPool();
            } else {
                ythrow yexception() << "No config found for a thread pool with a name " << poolName;
            }

            ThreadPools_[poolName] = pool;

            return pool;
        }
    }

    TSimpleSharedPtr<IThreadPool> operator[](TStringBuf poolName) override {
        return GetThreadPool(poolName);
    }

    TSimpleSharedPtr<IThreadPool> GetDefaultPool() override {
        return GetThreadPool(DefaultPoolMeta::Name);
    }

    size_t Size() const {
        return ThreadPools_.size();
    }

private:
    TSimpleSharedPtr<IThreadPool> CreateThreadPool(
            const TString& poolName,
            size_t threadsCount,
            size_t queueSizeLimit) const
    {
        auto statusListener = StatusListenerFactory_
            ? StatusListenerFactory_(poolName, {{"threadPool", poolName}})
            : nullptr;

        return MakeSimpleShared<TThreadPoolProxyWithImplementationOwning>(
                threadsCount,
                queueSizeLimit,
                std::move(statusListener));
    }

    TSimpleSharedPtr<IThreadPool> CreateThreadPool(const TThreadPoolConfig& poolConfig) const {
        return CreateThreadPool(poolConfig.GetName(), poolConfig.GetThreads(), poolConfig.GetMaxQueueSize());
    }

    TSimpleSharedPtr<IThreadPool> CreateDefaultThreadPool() const {
        return CreateThreadPool(DefaultPoolMeta::Name, DefaultPoolMeta::ThreadCount, DefaultPoolMeta::QueueSizeLimit);
    }

private:
    struct DefaultPoolMeta {
        static constexpr size_t ThreadCount = 4;
        static constexpr size_t QueueSizeLimit = 0; // unlimited
        static const TString Name;
    };

    TAdaptiveLock Lock_;
    THashMap<TString, TThreadPoolConfig> ThreadPoolConfigs_;
    THashMap<TString, TSimpleSharedPtr<IThreadPool>> ThreadPools_;
    TThreadPoolStatusListenerFactory StatusListenerFactory_;
};

const TString TLazyThreadPoolProvider::DefaultPoolMeta::Name = "Default";

} // namespace

IThreadPoolProviderPtr CreateLazyThreadPoolProvider(
    const TThreadPoolProviderConfig& config,
    TThreadPoolStatusListenerFactory listenerFactory
) {
    return {new TLazyThreadPoolProvider(config, std::move(listenerFactory))};
}

IThreadPoolProviderPtr CreateLazyThreadPoolProvider(TThreadPoolStatusListenerFactory listenerFactory) {
    return {new TLazyThreadPoolProvider(std::move(listenerFactory))};
}

} // namespace NSolomon::NAgent
