#include <solomon/agent/lib/thread/pool_provider.h>
#include <solomon/agent/lib/thread/thread_pool.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/yexception.h>

#include <google/protobuf/text_format.h>

using namespace NSolomon::NAgent;

class TConfigParseException: public yexception {};

void ParseTestConfig(TThreadPoolProviderConfig* config) {
    if (!google::protobuf::TextFormat::ParseFromString(NResource::Find("provider_ut.conf"), config)) {
        ythrow TConfigParseException() << "Failed to parse a provider config";
    }
}

void ParseConfigFromString(const TString& configString, TThreadPoolProviderConfig* config) {
    if (!google::protobuf::TextFormat::ParseFromString(configString, config)) {
        ythrow TConfigParseException() << "Failed to parse a provider config";
    }
}

void AssertPoolParameters(
        IThreadPoolProvider& poolProvider, TStringBuf poolName,
        size_t expectedThreadCount, size_t expectedQueueSize = 0)
{
    TSimpleSharedPtr<IThreadPool> pool = poolProvider.GetThreadPool(poolName);
    auto& p = static_cast<TThreadPoolProxyWithImplementationOwning&>(*pool);

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), expectedThreadCount);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), expectedThreadCount);

    if (expectedQueueSize) {
        ASSERT_EQ(p.InternalPool().GetMaxQueueSize(), expectedQueueSize);
    }
}

TEST(TThreadPoolProviderTest, ConstructingFromConfig) {
    TThreadPoolProviderConfig config;
    ParseTestConfig(&config);

    IThreadPoolProviderPtr poolProvider = CreateLazyThreadPoolProvider(config);

    AssertPoolParameters(*poolProvider, "CpuLowPriority", 30);
    AssertPoolParameters(*poolProvider, "CpuHighPriority", 5, 200);
    AssertPoolParameters(*poolProvider, "Io", 8);
    AssertPoolParameters(*poolProvider, "Scheduler", 4);
    AssertPoolParameters(*poolProvider, "ConfigDB", 5);
}

TEST(TThreadPoolProviderTest, NoConfig) {
    IThreadPoolProviderPtr poolProvider = CreateLazyThreadPoolProvider();

    TSimpleSharedPtr<IThreadPool> pool1 = poolProvider->GetThreadPool("Default");
    ASSERT_TRUE(pool1);

    TSimpleSharedPtr<IThreadPool> pool2 = poolProvider->GetDefaultPool();
    ASSERT_TRUE(pool2);

    ASSERT_EQ(pool1, pool2);
}

TEST(TThreadPoolProviderTest, RedefineDefaultPool) {
    TThreadPoolProviderConfig config;
    ParseConfigFromString(
        R"(
            ThreadPools: [
              {
                Name: "Default"
                Threads: 10
              }
            ]
        )",
        &config
    );

    IThreadPoolProviderPtr poolProvider = CreateLazyThreadPoolProvider(config);

    TSimpleSharedPtr<IThreadPool> pool = poolProvider->GetThreadPool("Default");
    AssertPoolParameters(*poolProvider, "Default", 10);
}

TEST(TThreadPoolTest, StartWithNoSideEffects) {
    TThreadPoolProxyWithImplementationOwning p(/*threads*/4, /*unlimited queue*/0);
    IThreadPool *pool = &p;

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 4u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 4u);

    pool->Start(1, 1);

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 4u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 4u);
}

TEST(TThreadPoolTest, StopWithNoSideEffects) {
    TThreadPoolProxyWithImplementationOwning p(/*threads*/4, /*unlimited queue*/0);
    IThreadPool *pool = &p;

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 4u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 4u);

    pool->Stop();

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 4u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 4u);
}

TEST(TThreadPoolTest, InheritedStopMethod) {
    TThreadPoolProxyWithImplementationOwning p(/*threads*/4, /*unlimited queue*/0);

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 4u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 4u);

    p.InternalPool().Stop();

    ASSERT_EQ(p.InternalPool().GetThreadCountExpected(), 0u);
    ASSERT_EQ(p.InternalPool().GetThreadCountReal(), 0u);
}
