#include "background_threads.h"

#include <util/system/thread.h>

namespace NSolomon {
namespace NAgent {

TBackgroundThreads::TBackgroundThreads()
    : KeepRunning_(1)
{
}

TBackgroundThreads::~TBackgroundThreads() {
    Stop();
}

void TBackgroundThreads::Stop() {
    // do nothing if this is called twice
    if (KeepRunning_.exchange(0, std::memory_order_relaxed)) {
        for (auto& t: Threads_) {
            t.join();
        }
    }
}

void TBackgroundThreads::RunInBackground(
        TStringBuf threadName,
        TCallable callable,
        TDuration initialSleep)
{
    auto method = &TBackgroundThreads::ExecuteInLoop;
    Threads_.emplace_back(method, this, threadName, callable, initialSleep);
}

void TBackgroundThreads::ExecuteInLoop(
        TStringBuf threadName,
        const TCallable& callable,
        TDuration initialSleep) const
{
    TThread::SetCurrentThreadName(threadName.data());

    IncrementalSleep(initialSleep);
    while (KeepRunning_.load(std::memory_order_relaxed)) {
        TDuration sleepTime = callable();
        if (sleepTime == TDuration::Max()) {
            break;
        }
        IncrementalSleep(sleepTime);
    }
}

void TBackgroundThreads::IncrementalSleep(TDuration amount) const {
    static const TDuration ONE_SECOND = TDuration::Seconds(1);

    while (amount > TDuration::Zero() && KeepRunning_.load(std::memory_order_relaxed)) {
        auto s = amount > ONE_SECOND ? ONE_SECOND : amount;
        Sleep(s);
        amount -= s;
    }
}

} // namespace NAgent
} // namespace NSolomon
