#pragma once

#include <util/generic/noncopyable.h>
#include <util/system/event.h>

#include <atomic>

namespace NSolomon::NAgent {

class TNopDelayPolicy {
public:
    static void RandomSleep() {
    }
};

/**
 * TODO: Could be largely improved. For comparison, look at
 * https://github.com/openjdk-mirror/jdk7u-jdk/blob/master/src/share/classes/java/util/concurrent/ThreadPoolExecutor.java
 * https://github.com/Microsoft/referencesource/blob/master/mscorlib/system/threading/CountdownEvent.cs
 */
template <class TPolicy = TNopDelayPolicy>
class TCountdownEventImpl: NNonCopyable::TMoveOnly {
public:
    TCountdownEventImpl() {
        Reset();
    }

    TCountdownEventImpl(TCountdownEventImpl&& other) {
        MoveFrom(std::move(other));
    }

    TCountdownEventImpl& operator=(TCountdownEventImpl&& other) {
        MoveFrom(std::move(other));
        return *this;
    }

    bool TryAdd(ui64 add) {
        Counter_ += add;

        TPolicy::RandomSleep();

        if (State_ != ECounterState::Running) {
            TPolicy::RandomSleep();

            Sub(add);
            return false;
        }

        return true;
    }

    bool TryInc() {
        return TryAdd(1);
    }

    void Sub(ui64 sub) {
        // May fire an extra signal, but Await will handle all of them because of the loop

        i64 newValue = Counter_.fetch_sub(sub) - sub;

        if (newValue == 0) {
            TPolicy::RandomSleep();

            if (IsSuspended()) {
                StopEvent_.Signal();
            }
        } else if (newValue < 0) {
            ythrow yexception() << "Counter's value dropped below zero";
        }
    }

    void Dec() {
        Sub(1);
    }

    // TODO: Could be added in an improved version
    // TODO: Value could (and will) be incorrect because of side-effects of TryAdd()
    /* ui64 Value() const { */
    /*     return AtomicGet(Counter_); */
    /* } */

    bool IsStopped() const {
        return State_.load(std::memory_order_relaxed) == ECounterState::Stopped;
    }

    void Stop() {
        State_ = ECounterState::Suspended;
    }

    bool Await(TDuration timeout = TDuration::Max()) {
        if (State_.load(std::memory_order_relaxed) == ECounterState::Stopped) {
            return true;
        }

        while (Counter_.load(std::memory_order_relaxed) != 0) {
            if (!StopEvent_.WaitT(timeout)) {
                return false;
            }
        }

        State_ = ECounterState::Stopped;
        return true;
    }

private:
    void Reset() {
        State_ = ECounterState::Running;
        Counter_ = 0;
        StopEvent_ = {};
    }

    void MoveFrom(TCountdownEventImpl&& other) {
        State_.exchange(other.State_);
        Counter_.exchange(other.Counter_);
        StopEvent_ = other.StopEvent_;

        other.Reset();
    }

    bool IsSuspended() const {
        return State_.load(std::memory_order_relaxed) == ECounterState::Suspended;
    }

private:
    enum class ECounterState {
        Running = 0,
        Suspended,
        Stopped,
    };

    std::atomic<ECounterState> State_;
    static_assert(decltype(State_)::is_always_lock_free == std::atomic<ui64>::is_always_lock_free);
    std::atomic<ui64> Counter_;
    TManualEvent StopEvent_;
};

using TCountdownEvent = TCountdownEventImpl<>;

} // namespace NSolomon::NAgent
