#pragma once

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

#include <array>

namespace NEnumDefineImpl {

template <typename E>
using IsEnum = typename std::enable_if<std::is_enum<E>::value>::type;

template <typename E, typename = IsEnum<E>>
constexpr static ui32 EnumSize() {
    return 0;
}

template <typename E, typename = IsEnum<E>>
inline static TStringBuf EnumValuesString() {
    return "";
}

} // namespace NEnumDefineImpl


template <typename EnumType, NEnumDefineImpl::IsEnum<EnumType>* = nullptr>
class TEnumInfo {
    TEnumInfo() = delete;

public:
    constexpr static ui32 Size() {
        return NEnumDefineImpl::EnumSize<EnumType>();
    }

    static auto const& ValuesStr() {
        static std::array<TStringBuf, Size()> values;
        if (values[0].empty()) {
            auto valuesStr = NEnumDefineImpl::EnumValuesString<EnumType>();
            for (auto& value: values) {
                while (valuesStr[0] == ' ') {
                    valuesStr.Skip(1);
                }
                if (!valuesStr.TrySplit(',', value, valuesStr)) {
                    value = valuesStr;
                }
            }
        }
        return values;
    }

    static const auto& Values() {
        static auto values = make_array(std::make_index_sequence<Size()>());
        return values;
    }

    constexpr static TStringBuf const& ToString(EnumType arg) {
        return ValuesStr()[static_cast<ui32>(arg)];
    }

    static EnumType FromString(TStringBuf val) {
        auto const& valuesStr = ValuesStr();
        for (ui32 i = 0; i < valuesStr.size(); i++) {
            if (val == valuesStr[i]) {
                return static_cast<EnumType>(i);
            }
        }
        ythrow yexception() << "Unknown enum value: " << val;
    }

private:
    template <std::size_t... Idx>
    static auto make_array(std::index_sequence<Idx...>) {
        return std::array<EnumType, Size()>{{ static_cast<EnumType>(Idx)... }};
    }
};

//
// NOTE: this macros works only for enums defined in global namespace
//
#define ENUM_DEFINE(EnumName, ...)\
    enum class EnumName;\
    \
    namespace NEnumDefineImpl {\
        template <>\
        constexpr ui32 EnumSize<EnumName>() {\
            enum EnumName { __VA_ARGS__ };\
            EnumName enumArray[]{ __VA_ARGS__ };\
            return sizeof(enumArray) / sizeof(enumArray[0]);\
        }\
        template <>\
        inline TStringBuf EnumValuesString<EnumName>() {\
            return #__VA_ARGS__;\
        }\
    }\
    template <>\
    void Out<EnumName>(IOutputStream& out, EnumName e) {\
        out << TEnumInfo<EnumName>::ToString(e);\
    }\
    \
    enum class EnumName { __VA_ARGS__ }
