#include "labels_match.h"

#include <solomon/libs/cpp/glob/glob.h>

namespace NSolomon::NAgent {

    using NMonitoring::ILabels;

    namespace {
        auto FindLabelByName(TStringBuf labelName, const ILabels& labels, const ILabels* optionalLabels) {
            if (auto usualLabel = labels.Get(labelName)) {
                return usualLabel;
            }

            return optionalLabels ? optionalLabels->Get(labelName) : std::nullopt;
        }
    } // namespace

    EMatchType MatchTypeFromStr(TStringBuf str) {
        if (str == TStringBuf("EXACT"))
            return EMatchType::EXACT;
        else if (str == TStringBuf("PREFIX"))
            return EMatchType::PREFIX;
        else if (str == TStringBuf("CONTAINS"))
            return EMatchType::CONTAINS;
        else if (str == TStringBuf("GLOB"))
            return EMatchType::GLOB;
        else if (str == TStringBuf("REGEXP"))
            return EMatchType::REGEXP;

        ythrow yexception() << "unknown match type: " << str;
    }

    TStringBuf MatchTypeToStr(EMatchType type) {
        switch (type) {
            case EMatchType::EXACT:
                return TStringBuf("EXACT");
            case EMatchType::PREFIX:
                return TStringBuf("PREFIX");
            case EMatchType::CONTAINS:
                return TStringBuf("CONTAINS");
            case EMatchType::GLOB:
                return TStringBuf("GLOB");
            case EMatchType::REGEXP:
                return TStringBuf("REGEXP");
        }
        Y_FAIL("invalid match type");
    }

    bool MatchExact(const ILabels& pattern, const ILabels& labels, const ILabels* optionalLabels) {
        for (auto&& patternLabel: pattern) {
            auto label = FindLabelByName(patternLabel.Name(), labels, optionalLabels);
            if (!label.has_value()) {
                return false;
            }

            if (label.value()->Value() != patternLabel.Value()) {
                return false;
            }
        }

        return true;
    }

    bool MatchContains(const ILabels& pattern, const ILabels& labels, const ILabels* optionalLabels) {
        for (auto&& patternLabel: pattern) {
            auto label = FindLabelByName(patternLabel.Name(), labels, optionalLabels);
            if (!label.has_value()) {
                return false;
            }

            if (!TStringBuf{label.value()->Value()}.Contains(patternLabel.Value())) {
                return false;
            }
        }
        return true;
    }

    bool MatchGlob(const ILabels& pattern, const ILabels& labels, const ILabels* optionalLabels) {
        // For now, only matches label values as GLOB. Label names are matched as EXACT
        for (auto&& patternLabel: pattern) {
            auto label = FindLabelByName(patternLabel.Name(), labels, optionalLabels);
            if (!label.has_value()) {
                return false;
            }

            if (!IsGlobMatch(patternLabel.Value(), label.value()->Value())) {
                return false;
            }
        }
        return true;
    }

    // TODO: implement other matchers

    TLabelsMatcher CreateMatcher(EMatchType matchType, const ILabels& pattern) {
        if (pattern.Empty()) {
            return [](const ILabels&, const ILabels* /*optionalLabels*/) { return true; };
        }

        switch (matchType) {
            case EMatchType::EXACT:
                return [&pattern](const ILabels& labels, const ILabels* optionalLabels) {
                    return MatchExact(pattern, labels, optionalLabels);
                };

            case EMatchType::CONTAINS:
                return [&pattern](const ILabels& labels, const ILabels* optionalLabels) {
                    return MatchContains(pattern, labels, optionalLabels);
                };

            case EMatchType::GLOB:
                return [&pattern](const ILabels& labels, const ILabels* optionalLabels) {
                    return MatchGlob(pattern, labels, optionalLabels);
                };

            default:
                ythrow yexception() << "unsupported labels match type: "
                                    << static_cast<int>(matchType);
        }
    }

}
