#include "logger.h"

#include <solomon/agent/misc/logger.h_serialized.h>
#include <solomon/agent/protos/logger_config.pb.h>

#include <library/cpp/logger/stream.h>
#include <library/cpp/logger/file.h>
#include <library/cpp/logger/system.h>
#include <library/cpp/logger/null.h>
#include <library/cpp/logger/filter.h>

#include <util/datetime/systime.h>
#include <util/generic/yexception.h>

#include <stdio.h>
#include <string.h>
#include <time.h>

namespace NSolomon {
namespace NAgent {
namespace {

ELogPriority LevelToPriority(ELevel level) {
    switch (level) {
    case ELevel::TRACE: return TLOG_RESOURCES;
    case ELevel::DEBUG: return TLOG_DEBUG;
    case ELevel::INFO: return TLOG_INFO;
    case ELevel::WARN: return TLOG_WARNING;
    case ELevel::ERROR: return TLOG_ERR;
    case ELevel::FATAL: return TLOG_EMERG;
    default:
        ythrow yexception() << "unknown log level: " << ELevel_Name(level);
    }
}

ELogLevel ConvertLevel(ELevel level) {
    switch (level) {
    case ELevel::TRACE: return ELogLevel::TRACE;
    case ELevel::DEBUG: return ELogLevel::DEBUG;
    case ELevel::INFO: return ELogLevel::INFO;
    case ELevel::WARN: return ELogLevel::WARN;
    case ELevel::ERROR: return ELogLevel::ERROR;
    case ELevel::FATAL: return ELogLevel::FATAL;
    default:
        ythrow yexception() << "unknown log level: " << ELevel_Name(level);
    }
}

TStringBuf LevelToString(ELogLevel level) {
    switch (level) {
    case ELogLevel::FATAL: return TStringBuf("FATAL");
    case ELogLevel::ERROR: return TStringBuf("ERROR");
    case ELogLevel::WARN : return TStringBuf("WARN ");
    case ELogLevel::INFO : return TStringBuf("INFO ");
    case ELogLevel::DEBUG: return TStringBuf("DEBUG");
    case ELogLevel::TRACE: return TStringBuf("TRACE");
    }
}

THolder<TLogBackend> CreateLogBackend(const TLoggerConfig &config) {
    switch (config.GetLogTo()) {
    case ELogTo::FILE:
        return MakeHolder<TFileLogBackend>(config.GetLogFile());
    case ELogTo::SYSLOG: {
        auto facility = TSysLogBackend::TSYSLOG_LOCAL1;
        return MakeHolder<TSysLogBackend>("solomon-agent", facility);
    }
    default:
        return MakeHolder<TStreamLogBackend>(&Cerr);
    }
}

void WriteLocalTime(IOutputStream* out) {
    struct timeval now;
    gettimeofday(&now, nullptr);

    struct tm tm;
    time_t seconds = static_cast<time_t>(now.tv_sec);
    localtime_r(&seconds, &tm);

    char buf[sizeof("2016-01-02 03:04:05.006")];
    int n = strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S.", &tm);
    snprintf(buf + n, sizeof(buf) - n, "%03u", static_cast<ui32>(now.tv_usec) / 1000);

    out->Write(buf, sizeof(buf) - 1);
}

const char* Basename(const char* name) {
    const char* p = name ? std::strrchr(name, LOCSLASH_C) : nullptr;
    return p ? p + 1 : name;
}

class TSolomonLogElement: public TLogElement {
public:
    explicit TSolomonLogElement(const TLog* parent)
        : TLogElement(parent)
    {
    }

    ~TSolomonLogElement() {
        *this << '\n';
    }
};


} // namespace


TSolomonLog::TSolomonLog()
    : TLog(MakeHolder<TNullLogBackend>())
{
}

TSolomonLog::TSolomonLog(THolder<TLogBackend> backend)
    : TLog(std::move(backend))
{
}

void TSolomonLog::EnableEntryCounting(NMonitoring::TMetricRegistry* metricRegistry) {
    Registry_ = metricRegistry;

    LevelMetrics_.reserve(GetEnumItemsCount<ELogLevel>());
    for (size_t i = 0; i < GetEnumItemsCount<ELogLevel>(); ++i) {
        NMonitoring::TLabels labels{
            {"sensor", "logger.entryCount"},
            {"logLevel", LevelToString(static_cast<ELogLevel>(i))}
        };

        auto metric = Registry_->Rate(std::move(labels));
        LevelMetrics_.emplace_back(metric);
    }
}

THolder<TLogElement> TSolomonLog::CreateLogElement(
        ELogLevel level, const char* file, int line)
{
    auto element = MakeHolder<TSolomonLogElement>(this);
    WriteLocalTime(element.Get());
    *element
        << ' '<< LevelToString(level) << TStringBuf(" {")
        << Basename(file) << ':' << line << TStringBuf("}: ");

    if (!LevelMetrics_.empty()) {
        LevelMetrics_[static_cast<size_t>(level)]->Inc();
    }

    return THolder(element.Release());
}

void InitLogger(
        const TLoggerConfig& config,
        NMonitoring::TMetricRegistry* metricRegistry)
{
    THolder<TLogBackend> backend = CreateLogBackend(config);
    ELogPriority priority = LevelToPriority(config.GetLevel());

    if (priority != LOG_MAX_PRIORITY) {
        backend.Reset(new TFilteredLogBackend(
                          std::move(backend), priority));
    }

    GetLogger().ResetBackend(std::move(backend));
    GetLogger().SetDefaultPriority(priority);
    GetLogger().SetCurrentLevel(ConvertLevel(config.GetLevel()));
    if (metricRegistry) {
        GetLogger().EnableEntryCounting(metricRegistry);
    }
}

void InitLogger(
        IOutputStream* out,
        NMonitoring::TMetricRegistry* metricRegistry)
{
    GetLogger().ResetBackend(MakeHolder<TStreamLogBackend>(out));
    GetLogger().SetDefaultPriority(TLOG_INFO);
    GetLogger().SetCurrentLevel(ELogLevel::INFO);
    if (metricRegistry) {
        GetLogger().EnableEntryCounting(metricRegistry);
    }
}

} // namespace NAgent
} // namespace NSolomon
