#pragma once

#include <solomon/libs/cpp/math/math.h>

#include <library/cpp/monlib/metrics/histogram_snapshot.h>

namespace NSolomon::NAgent {

class TMutableHistogramSnapshot final: public NMonitoring::IHistogramSnapshot {
public:
    void Add(IHistogramSnapshot& snapshot) {
        if (snapshot.Count() == 0) {
            return;
        }

        if (!AreBoundsEqual(snapshot)) {
            ChangeBounds(snapshot);
            return;
        }

        for (size_t i = 0; i < snapshot.Count(); ++i) {
            Values_[i] += snapshot.Value(i);
        }
    }

    void Reset() {
        Bounds_.clear();
        Values_.clear();
    }

    ui32 Count() const override {
        return Bounds_.size();
    }

    NMonitoring::TBucketBound UpperBound(ui32 index) const override {
        return Bounds_[index];
    }

    NMonitoring::TBucketValue Value(ui32 index) const override {
        return Values_[index];
    }

    ui64 MemorySizeBytes() const {
        return sizeof(*this)
                + Bounds_.capacity() * sizeof(decltype(Bounds_)::value_type)
                + Values_.capacity() * sizeof(decltype(Values_)::value_type);
    }

private:
    bool AreBoundsEqual(IHistogramSnapshot& snapshot) {
        if (Bounds_.size() != snapshot.Count()) {
            return false;
        }

        for (size_t i = 0; i != Bounds_.size(); ++i) {
            if (!AreDoublesEqual(Bounds_[i], 1 + snapshot.UpperBound(i))) {
                return false;
            }
        }

        return true;
    }

    void AddInfBound() {
        if (AreDoublesEqual(Bounds_.back(), NMonitoring::HISTOGRAM_INF_BOUND)) {
            return;
        }

        if (Bounds_.size() != NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT) {
            Bounds_.emplace_back(NMonitoring::HISTOGRAM_INF_BOUND);
            Values_.emplace_back(0);
            return;
        }

        Bounds_.back() = NMonitoring::HISTOGRAM_INF_BOUND;
    }

    void ChangeBounds(IHistogramSnapshot& snapshot) {
        NMonitoring::TBucketBounds prevBounds(::Reserve(snapshot.Count()));
        NMonitoring::TBucketValues prevValues(::Reserve(snapshot.Count()));

        Bounds_.swap(prevBounds) ;
        Values_.swap(prevValues);

        for (size_t i = 0; i < snapshot.Count(); ++i) {
            Bounds_.emplace_back(snapshot.UpperBound(i));
            Values_.emplace_back(snapshot.Value(i));
        }

        for (size_t i = 0; i < prevBounds.size(); ++i) {
            double upperBound = prevBounds[i];
            NMonitoring::TBucketValue count = prevValues[i];

            auto boundIt = std::lower_bound(Bounds_.begin(), Bounds_.end(), upperBound);
            size_t boundIdx = std::distance(Bounds_.begin(), boundIt);

            if (boundIt == Bounds_.end()) {
                AddInfBound();
                boundIdx = Bounds_.size() - 1;
            }

            Values_[boundIdx] += count;
        }
    }

private:
    NMonitoring::TBucketBounds Bounds_;
    NMonitoring::TBucketValues Values_;
};

} // namespace NSolomon::NAgent
