#pragma once

#include <solomon/agent/lib/thread/pool_provider.h>
#include <solomon/agent/misc/background_threads.h>

#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/system/guard.h>
#include <util/system/spinlock.h>

namespace NSolomon::NAgent {

/**
 * Dispatches all registered functions every specified interval
 */
class TTimerDispatcher: public TAtomicRefCount<TTimerDispatcher> {
public:
    using TFnKey = ui64;
    using TFunction = std::function<void(TInstant)>;

    class TFunctionHolder: public NNonCopyable::TMoveOnly {
    public:
        ~TFunctionHolder() {
            Release();
        }

        TFunctionHolder()
            : TimerDispatcher_{nullptr}
        {}

        TFunctionHolder(TTimerDispatcher* timerDispatcher, TFnKey fnKey)
            : TimerDispatcher_{timerDispatcher}
            , FnKey_{fnKey}
        {
        }

        TFunctionHolder(TFunctionHolder&& other) {
            MoveFrom(std::move(other));
        }

        TFunctionHolder& operator=(TFunctionHolder&& other) {
            MoveFrom(std::move(other));
            return *this;
        }

        void Release() {
            if (TimerDispatcher_) {
                TimerDispatcher_->Unregister(FnKey_);
                TimerDispatcher_ = nullptr;
            }
        }

    private:
        void MoveFrom(TFunctionHolder&& other) {
            TimerDispatcher_= other.TimerDispatcher_;
            FnKey_ = other.FnKey_;
            other.TimerDispatcher_ = nullptr;
        }

    private:
        TTimerDispatcher* TimerDispatcher_;
        TFnKey FnKey_;
    };

public:
    ~TTimerDispatcher() {
        try {
            BackgroundThread_.Stop();
        } catch (...) {
        }
    }

    // TODO: pass the pool?
    TTimerDispatcher(const TDuration& dispatchInterval)
        : DispatchInterval_{dispatchInterval}
    {
        ui64 intervalMs = dispatchInterval.MilliSeconds();

        TInstant initTime = TInstant::Now();
        ui64 initMs = initTime.MilliSeconds();
        CurrentIntervalStart_ = TInstant::MilliSeconds(initMs - (initMs % intervalMs));
        TInstant expectedEnd = CurrentIntervalStart_ + DispatchInterval_;

        TDuration initialSleep = expectedEnd - initTime;

        BackgroundThread_.RunInBackground(
            "TTimerDispatcher",
            [this, intervalMs]() -> TDuration {
                TInstant now = TInstant::Now();
                ui64 tsMs = now.MilliSeconds();

                CurrentIntervalStart_ = TInstant::MilliSeconds(tsMs - (tsMs % intervalMs));

                with_lock (FunctionsLock_) {
                    for (const auto& it: FunctionsToDispatch_) {
                        // Telling about the interval that's coming next
                        it.second(CurrentIntervalStart_);
                    }
                }

                TInstant expectedEnd = CurrentIntervalStart_ + DispatchInterval_;
                now = TInstant::Now();

                TDuration toSleep = (now < expectedEnd) ? (expectedEnd - now) : TDuration::Zero();
                return toSleep;
            },
            initialSleep
        );
    }

    TFunctionHolder RegisterWithOwning(TFunction&& fn) {
        auto fnKey = RegisterSilently(std::move(fn));
        return TFunctionHolder{this, fnKey};
    }

    TFnKey RegisterSilently(TFunction&& fn) {
        TFnKey cntValue = FnCnt_.fetch_add(1, std::memory_order_relaxed);

        with_lock (FunctionsLock_) {
            auto [_, isInserted] = FunctionsToDispatch_.emplace(cntValue, fn);
            Y_ENSURE(isInserted, "Failed to register a function");
        }

        return cntValue;
    }

    void Unregister(TFnKey fnKey) {
        with_lock (FunctionsLock_) {
            FunctionsToDispatch_.erase(fnKey);
        }
    }

    TDuration DispatchInterval() {
        return DispatchInterval_;
    }

    TInstant CurrentIntervalStart() {
        return CurrentIntervalStart_;
    }

private:
    TBackgroundThreads BackgroundThread_;
    TInstant CurrentIntervalStart_;
    TDuration DispatchInterval_;
    std::atomic<ui64> FnCnt_{0};
    TAdaptiveLock FunctionsLock_;
    THashMap<TFnKey, TFunction> FunctionsToDispatch_;
};

using TTimerDispatcherPtr = TIntrusivePtr<TTimerDispatcher>;

} // namespace NSolomon::NAgent
