#include "timer_thread.h"

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>
#include <util/string/cast.h>
#include <util/system/condvar.h>
#include <util/system/mutex.h>
#include <util/system/thread.h>

namespace {

static std::atomic<ui64> NextSerialNumber = 0;

struct TTimerTaskLess {
    bool operator()(const ITimerTaskPtr& a, const ITimerTaskPtr& b) {
        return a->NextExecutionTime() > b->NextExecutionTime();
    }
};

} // namespace

///////////////////////////////////////////////////////////////////////////////
// TTaskQueue
///////////////////////////////////////////////////////////////////////////////
class TTaskQueue {
public:
    void Add(ITimerTaskPtr task, bool signal) {
        with_lock (Mutex_) {
            Tasks_.push_back(std::move(task));
            PushHeap(Tasks_.begin(), Tasks_.end(), TTimerTaskLess());

            if (signal) {
                CondVar_.Signal(); // wakeup possibly waiting thread
            }
        }
    }

    ITimerTaskPtr PopReady(TInstant currentTime) {
        with_lock (Mutex_) {
            if (Tasks_.empty()) {
                // wait some time to avoid busy loop
                CondVar_.WaitT(Mutex_, TDuration::Seconds(5));
            }

            if (!Tasks_.empty()) {
                TInstant executionTime = Tasks_.front()->NextExecutionTime();
                if (executionTime <= currentTime) {
                    // task is ready to be executed
                    PopHeap(Tasks_.begin(), Tasks_.end(), TTimerTaskLess());
                    ITimerTaskPtr task = Tasks_.back();
                    Tasks_.pop_back();
                    return task;
                } else {
                    // wait till task will be ready or new task is submitted
                    CondVar_.WaitT(Mutex_, executionTime - currentTime);
                }
            }

            return nullptr;
        }
    }

    void CancellAll() {
        with_lock (Mutex_) {
            for (const ITimerTaskPtr& task: Tasks_) {
                task->Cancel();
            }
            Tasks_.clear();
            CondVar_.Signal(); // wakeup possibly waiting thread
        }
    }

    size_t PurgeCancelled() {
        size_t result = 0;

        with_lock (Mutex_) {
            for (size_t i = Tasks_.size() - 1; i > 0; i--) {
                if (Tasks_[i]->State() == TTaskState::CANCELLED) {
                    Tasks_[i] = Tasks_.back();
                    Tasks_.pop_back();
                    result++;
                }
            }

            if (result != 0) {
                MakeHeap(Tasks_.begin(), Tasks_.end(), TTimerTaskLess());
            }
        }

        return result;
    }

private:
    TVector<ITimerTaskPtr> Tasks_;
    TMutex Mutex_;
    TCondVar CondVar_;
};


///////////////////////////////////////////////////////////////////////////////
// TTimerThread
///////////////////////////////////////////////////////////////////////////////
TTimerThread::TTimerThread(TStringBuf name)
    : Name_(name.empty() ? TStringBuf("Timer-") : name)
    , TaskQueue_(new TTaskQueue)
{
    if (name.empty()) {
        Name_ += ToString(NextSerialNumber.fetch_add(1) + 1);
    }
}

TTimerThread::~TTimerThread() {
    Stop();
}

void TTimerThread::Start() {
    if (!NewTasksMayBeScheduled_.exchange(1, std::memory_order_relaxed)) {
        Thread_.Reset(new std::thread(&TTimerThread::MainLoop, this));
    }
}

void TTimerThread::Stop() {
    if (NewTasksMayBeScheduled_.exchange(0)) {
        TaskQueue_->CancellAll();
        if (Thread_->joinable()) {
            Thread_->join();
        }
        Thread_.Destroy();
    }
}

size_t TTimerThread::PurgeCancelled() {
    return TaskQueue_->PurgeCancelled();
}

void TTimerThread::DoSchedule(ITimerTaskPtr task, TInstant time, TDuration period) {
    if (!NewTasksMayBeScheduled_.load(std::memory_order_relaxed)) {
        ythrow yexception() << "Timer is stopped";
    }

    if (!task->Schedule(time, period)) {
        ythrow yexception() << "Task already scheduled or cancelled";
    }

    TaskQueue_->Add(std::move(task), true);
}

void TTimerThread::MainLoop() {
    TThread::SetCurrentThreadName(Name_.data());

    while (NewTasksMayBeScheduled_.load(std::memory_order_relaxed)) {
        try {
            ITimerTaskPtr task = TaskQueue_->PopReady(TInstant::Now());
            if (!task) {
                // no ready task in queue; go to check while-loop condition
                continue;
            } else if (task->State() == TTaskState::CANCELLED) {
                // task already cancelled, simply ignore it
                continue;
            }

            if (task->Period() != TDuration::Zero()) {
                // reschedule repeating task
                TInstant executionTime = task->NextExecutionTime();
                task->SetNextExecutionTime(executionTime + task->Period());
                TaskQueue_->Add(task, false);
            }

            // execute ready task if it is yet not cancelled
            task->Execute();
        } catch(...) {
            Cerr << "Unhandled exception while timer task execution: "
                 << CurrentExceptionMessage() << Endl;
        }
    }
}
