#pragma once

#include "task_state.h"

#include <util/datetime/base.h>
#include <util/generic/noncopyable.h>
#include <util/generic/ptr.h>
#include <util/generic/string.h>
#include <library/cpp/deprecated/atomic/atomic.h>
#include <util/thread/pool.h>

#include <thread>

// TODO: wrap it with the NSolomon::NAgent namespace

///////////////////////////////////////////////////////////////////////////////
// ITimerTask
///////////////////////////////////////////////////////////////////////////////
class ITimerTask: public TAtomicRefCount<ITimerTask> {
    friend class TTimerThread;

public:
    ITimerTask() {
        State_.store(TTaskState::VIRGIN, std::memory_order_relaxed);
    }

    virtual ~ITimerTask() = default;

    // Avoid doing heavy things in implementation of this function.
    // Because it will hang TimerThread.
    // Use TFuncTimerTask instead.
    virtual void Run() = 0;

    void Cancel() noexcept {
        State_.store(TTaskState::CANCELLED, std::memory_order_relaxed);
    }

    TTaskState::EState State() const noexcept {
        return State_.load(std::memory_order_relaxed);
    }

    TInstant NextExecutionTime() const noexcept {
        return NextExecutionTime_;
    }

    TDuration Period() const noexcept {
        return Period_;
    }

private:
    void SetNextExecutionTime(TInstant newTime) noexcept {
        NextExecutionTime_ = newTime;
    }

    bool Schedule(TInstant time, TDuration period) noexcept {
        auto expectedState = TTaskState::VIRGIN;
        if (State_.compare_exchange_strong(expectedState, TTaskState::SCHEDULED, std::memory_order_acq_rel)) {
            NextExecutionTime_ = time;
            Period_ = period;
            return true;
        }
        return false;
    }

    void Execute() {
        if (State_.exchange(TTaskState::EXECUTED) != TTaskState::CANCELLED) {
            Run();
        }
    }

private:
    std::atomic<TTaskState::EState> State_;
    TInstant NextExecutionTime_;
    TDuration Period_;
};

using ITimerTaskPtr = TIntrusivePtr<ITimerTask>;

struct TTaskContext {
    TInstant Scheduled;
};

///////////////////////////////////////////////////////////////////////////////
// TFuncTimerTask (runs given func in MtpQueue)
///////////////////////////////////////////////////////////////////////////////
template <typename TFunc>
class TFuncTimerTask final: public ITimerTask {
    using TFuncPtr = TAtomicSharedPtr<TFunc>;

    class TQueueFuncObj final: public IObjectInQueue {
    public:
        static TQueueFuncObj* Create(TFuncPtr func) {
            return new TQueueFuncObj{std::move(func)};
        }

        void Process(void*) override {
            try {
                if constexpr (std::is_invocable<TFunc, TTaskContext>::value) {
                    (*Func_)(TaskContext_);
                } else {
                    (*Func_)();
                }
            } catch (...) {
            }

            Dispose();
        }

    private:
        TQueueFuncObj(TFuncPtr func)
            : Func_{std::move(func)}
            , TaskContext_{ TInstant::Now() }
        {
        }

        void Dispose() noexcept {
            delete this;
        }

    private:
        TFuncPtr Func_;
        TTaskContext TaskContext_;
    };

public:
    TFuncTimerTask(IThreadPool* queue, TFunc&& func)
        : Queue_(queue)
        , Func_(new TFunc{std::move(func)})
    {
    }

    void Run() override {
        Queue_->SafeAdd(TQueueFuncObj::Create(Func_));
    }

private:
    IThreadPool* Queue_;
    TFuncPtr Func_;
};

template <typename TFunc>
ITimerTaskPtr MakeFuncTimerTask(IThreadPool* queue, TFunc&& func) {
    return new TFuncTimerTask<TFunc>(queue, std::move(func));
}


///////////////////////////////////////////////////////////////////////////////
// TTimerThread
///////////////////////////////////////////////////////////////////////////////
class TTaskQueue;
class TTimerThread: private TNonCopyable {
public:
    explicit TTimerThread(TStringBuf name = {});
    ~TTimerThread();

    void Schedule(ITimerTaskPtr task, TDuration delay) {
        DoSchedule(std::move(task), delay.ToDeadLine(), TDuration::Zero());
    }

    void Schedule(ITimerTaskPtr task, TInstant time) {
        DoSchedule(std::move(task), time, TDuration::Zero());
    }

    void Schedule(ITimerTaskPtr task, TDuration delay, TDuration period) {
        Y_ENSURE(period > TDuration::Zero());
        DoSchedule(std::move(task), delay.ToDeadLine(), period);
    }

    void Schedule(ITimerTaskPtr task, TInstant firstTime, TDuration period) {
        Y_ENSURE(period > TDuration::Zero());
        DoSchedule(std::move(task), firstTime, period);
    }

    void Start();
    void Stop();

    bool IsStarted() const {
        return static_cast<bool>(Thread_);
    }

    size_t PurgeCancelled();

    const TString& Name() const noexcept {
        return Name_;
    }

private:
    void DoSchedule(ITimerTaskPtr task, TInstant time, TDuration period);
    void MainLoop();

private:
    TString Name_;
    std::atomic<bool> NewTasksMayBeScheduled_{false};
    THolder<TTaskQueue> TaskQueue_;
    THolder<std::thread> Thread_;
};
