#include <solomon/agent/misc/timer_thread.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/vector.h>

class TFlagTask: public ITimerTask {
public:
    TFlagTask(std::atomic<bool>* executed)
        : Executed_(executed)
    {
    }

    void Run() override {
        *Executed_ = true;
    }

private:
    std::atomic<bool>* Executed_;
};

class TCountTask: public ITimerTask {
public:
    TCountTask(std::atomic<ui64>* counter)
        : Counter_(counter)
    {
    }

    void Run() override {
        ++(*Counter_);
    }

private:
    std::atomic<ui64>* Counter_;
};

class TAppendTask: public ITimerTask {
public:
    TAppendTask(TVector<ui32>* vector, ui32 value)
        : Vector_(vector)
        , Value_(value)
    {
    }

    void Run() override {
        Vector_->push_back(Value_);
    }

private:
    TVector<ui32>* Vector_;
    ui32 Value_;
};

TEST(TTimerThreadTest, Name) {
    TTimerThread timer1;
    ASSERT_EQ(timer1.Name(), "Timer-1");

    TTimerThread timer2;
    ASSERT_EQ(timer2.Name(), "Timer-2");

    TTimerThread timer("MyTimer");
    ASSERT_EQ(timer.Name(), "MyTimer");
}

TEST(TTimerThreadTest, ScheduleOnce) {
    TTimerThread timer;
    timer.Start();

    std::atomic<bool> executed{false};

    ITimerTaskPtr task = new TFlagTask(&executed);
    ASSERT_TRUE(task->State() == TTaskState::VIRGIN);

    timer.Schedule(task, TDuration::MilliSeconds(100));
    ASSERT_TRUE(task->State() == TTaskState::SCHEDULED);
    ASSERT_TRUE(executed == false);

    Sleep(TDuration::MilliSeconds(500));

    ASSERT_TRUE(task->State() == TTaskState::EXECUTED);
    ASSERT_TRUE(executed == true);
}

TEST(TTimerThreadTest, SchedulePeriodically) {
    TTimerThread timer;
    timer.Start();

    std::atomic<ui64> counter{0};

    ITimerTaskPtr task = new TCountTask(&counter);
    ASSERT_TRUE(task->State() == TTaskState::VIRGIN);

    timer.Schedule(task, TDuration::MilliSeconds(100), TDuration::MilliSeconds(10));
    ASSERT_TRUE(task->State() == TTaskState::SCHEDULED);
    ASSERT_TRUE(counter == 0);

    Sleep(TDuration::MilliSeconds(500));

    ASSERT_TRUE(task->State() == TTaskState::EXECUTED);
    // task executed about 400ms with 10ms period (aproximately 40 times)
    ui32 counterValue = counter;
    ASSERT_TRUE(counterValue >= 30 && counterValue <= 50);

    // cancel the task before stopping and destroying timer
    task->Cancel();
    timer.Stop();
}

TEST(TTimerThreadTest, Cancel) {
    TTimerThread timer;
    timer.Start();

    std::atomic<bool> executed{false};

    ITimerTaskPtr task = new TFlagTask(&executed);
    ASSERT_TRUE(task->State() == TTaskState::VIRGIN);

    timer.Schedule(task, TDuration::MilliSeconds(100));
    ASSERT_TRUE(task->State() == TTaskState::SCHEDULED);
    ASSERT_TRUE(executed == false);

    task->Cancel();
    ASSERT_TRUE(task->State() == TTaskState::CANCELLED);
    ASSERT_TRUE(executed == false);

    Sleep(TDuration::MilliSeconds(500));

    ASSERT_TRUE(task->State() == TTaskState::CANCELLED);
    ASSERT_TRUE(executed == false);
}

TEST(TTimerThreadTest, PurgeCancelled) {
    TTimerThread timer;
    timer.Start();

    std::atomic<bool> executed1{false};
    std::atomic<bool> executed2{false};

    ITimerTaskPtr task1 = new TFlagTask(&executed1);
    timer.Schedule(task1, TDuration::MilliSeconds(10));

    ITimerTaskPtr task2 = new TFlagTask(&executed2);
    timer.Schedule(task2, TDuration::MilliSeconds(20));

    task2->Cancel();

    ASSERT_TRUE(timer.PurgeCancelled() == 1);
    ASSERT_TRUE(timer.PurgeCancelled() == 0);

    Sleep(TDuration::MilliSeconds(500));

    ASSERT_TRUE(task1->State() == TTaskState::EXECUTED);
    ASSERT_TRUE(executed1 == true);

    ASSERT_TRUE(task2->State() == TTaskState::CANCELLED);
    ASSERT_TRUE(executed2 == false);
}

TEST(TTimerThreadTest, ExecutionOrder) {
    TTimerThread timer;
    timer.Start();

    TVector<ui32> values;
    timer.Schedule(new TAppendTask(&values, 3), TDuration::MilliSeconds(25));
    timer.Schedule(new TAppendTask(&values, 4), TDuration::MilliSeconds(30));
    timer.Schedule(new TAppendTask(&values, 1), TDuration::MilliSeconds(10));
    timer.Schedule(new TAppendTask(&values, 2), TDuration::MilliSeconds(20));

    Sleep(TDuration::MilliSeconds(200));

    timer.Stop();

    Sleep(TDuration::MilliSeconds(200));

    ASSERT_EQ(values, TVector<ui32>({1, 2, 3, 4}));
}

TEST(TTimerThreadTest, CancelAfterStop) {
    TTimerThread timer;
    timer.Start();

    std::atomic<bool> executed{false};

    ITimerTaskPtr task = new TFlagTask(&executed);
    ASSERT_TRUE(task->State() == TTaskState::VIRGIN);

    timer.Schedule(task, TDuration::Seconds(1));
    ASSERT_TRUE(task->State() == TTaskState::SCHEDULED);
    ASSERT_TRUE(executed == false);

    timer.Stop();

    ASSERT_TRUE(task->State() == TTaskState::CANCELLED);
    ASSERT_TRUE(executed == false);
}

TEST(TTimerThreadTest, FuncTasks) {
    TTimerThread timer;
    timer.Start();

    TThreadPool queue;
    queue.Start(2);

    std::atomic<bool> executed1{false};

    auto task = MakeFuncTimerTask(&queue, [&executed1]() {
        // emulate heavy computation
        Sleep(TDuration::MilliSeconds(500));
        executed1 = true;
    });
    timer.Schedule(task, TDuration::MilliSeconds(50));

    std::atomic<bool> executed2{false};

    timer.Schedule(new TFlagTask(&executed2), TDuration::MilliSeconds(60));

    Sleep(TDuration::MilliSeconds(200));

    ASSERT_TRUE(executed1 == false);
    ASSERT_TRUE(executed2 == true);

    Sleep(TDuration::Seconds(1));

    ASSERT_TRUE(executed1 == true);
}
