#include "decoder.h"
#include "caching_converter.h"

#include <solomon/agent/modules/agent/graphite/protos/graphite_config.pb.h>

#include <library/cpp/monlib/metrics/metric_consumer.h>

#include <util/charset/unidata.h>
#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/string/builder.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/string/subst.h>

#include <contrib/libs/re2/re2/re2.h>
#include <contrib/libs/re2/re2/set.h>

#include <array>

using namespace NMonitoring;


namespace NSolomon {
namespace NAgent {

namespace {

re2::StringPiece Sp(TStringBuf sb) {
    return {sb.data(), sb.size()};
}

constexpr auto WILD_LIMIT = 10;

THolder<re2::RE2> CompileRE2WithCheck(const std::string& pattern) {
    THolder<re2::RE2> re(new re2::RE2(pattern));
    Y_ENSURE(re->ok(), "Unable to compile regex " << pattern << ": " << re->error());

    return re;
}

#define ALLOWED_CHARS "[a-zA-Z0-9_\\-]"
#define ALLOWED_DOT "[a-zA-Z0-9_\\-\\.]"

const std::string PART_RE_STR = "([a-zA-Z0-9_]" ALLOWED_CHARS "*)";
THolder<re2::RE2> SIMPLE_PART_RE = CompileRE2WithCheck(PART_RE_STR);
THolder<re2::RE2> SIMPLE_REF_RE = CompileRE2WithCheck(R"(\$\d{1,2})");
THolder<re2::RE2> PATTERN_RE = CompileRE2WithCheck("((\\*|" ALLOWED_CHARS "+)\\.)*(\\*|" ALLOWED_CHARS "+)");
THolder<re2::RE2> LABEL_RE = CompileRE2WithCheck(ALLOWED_DOT "+");
THolder<re2::RE2> TEMPLATE_RE = CompileRE2WithCheck(R"(([a-zA-Z0-9]|\$\d{1,2})()" ALLOWED_DOT R"(|\$\d{1,2})*)");

#undef ALLOWED_DOT
#undef ALLOWED_CHARS

constexpr auto NO_REPLACE_IDX = 0;
constexpr auto SINGLE_REPLACE_IDX = 1;

template <typename TOnIdx, typename TOnString>
void CapturePlaceholders(TStringBuf str, TOnIdx onIdx, TOnString onString) {
    const auto len = str.size();

    size_t i = 0;
    size_t skip = 0;
    while (i < len) {
        TStringBuf suffix{str};
        suffix.Skip(i);

        skip = suffix.find('$');
        i += skip;

        if (skip == TString::npos) {
            onString(suffix);
            break;
        } else {
            onString(TStringBuf{suffix}.Head(skip));
        }

        ++i;

        const auto digitBegin = i;

        while (i < len && ::IsDigit(str[i])) {
            ++i;
        }

        const auto idx = ::FromString<long>(TStringBuf{str.data() + digitBegin, str.data() + i});
        onIdx(idx);
    }
};

long MaxPlaceholderIdx(TStringBuf str) {
    long result = -1;

    CapturePlaceholders(str,
        [&] (auto idx) { result = Max<decltype(result)>(idx, result); },
        [] (auto) {}
    );

    return result;
}

/// Fills out placeholders of form $\d{1,2} with corresponding values passed as
/// a second argument. No bound checking is performed.
TString FillPlaceholders(TStringBuf str, const TVector<TStringBuf>& vals) {
    TStringBuilder result;

    CapturePlaceholders(str,
        [&] (auto idx) { result << vals[idx - 1]; },
        [&] (auto sb) { result << sb; }
    );

    return result;
}

size_t NormalizePattern(TString& pattern) {
    auto wildCount = SubstGlobal(pattern, TStringBuf("*"), PART_RE_STR);
    Y_ENSURE(wildCount <= WILD_LIMIT, "Too many wildcards in the pattern");

    SubstGlobal(pattern, TStringBuf("."), TStringBuf("\\."));
    return wildCount;
}

class TGraphiteBlacklist {
public:
    TGraphiteBlacklist(const TGraphiteConversionConfig& config)
        : Blacklist_{re2::RE2::DefaultOptions, re2::RE2::ANCHOR_BOTH}
    {
        for (TString pattern: config.GetBlacklist().GetPattern()) {
            Y_ENSURE(RE2::FullMatch({pattern}, *PATTERN_RE), "Pattern " << pattern << " is illformated");
            NormalizePattern(pattern);

            std::string errorMsg;
            i64 reIdx = Blacklist_.Add({pattern}, &errorMsg);

            if (reIdx == -1) {
                ythrow yexception() << "Failed to add a pattern " << pattern << " to a set: " << errorMsg;
            }
        }

        Y_ENSURE(Blacklist_.Compile(), "Unable to compile blacklist regex");
    }

    bool Contains(TStringBuf name) const {
        return Blacklist_.Match({static_cast<std::string>(name)}, nullptr);
    }

private:
    re2::RE2::Set Blacklist_;
};

} // namespace

class TGraphiteMetricMap: public ILabelConverter {
public:
    explicit TGraphiteMetricMap(const TGraphiteConversionConfig& config)
        : Mode_{config.GetMode()}
        , AnyMatch_{re2::RE2::DefaultOptions, re2::RE2::ANCHOR_BOTH}
    {
        for (const auto& mapping: config.GetMetrics()) {
            Add(mapping);
        }

        if (!AnyMatch_.Compile()) {
            ythrow yexception() << "Set matcher ran out of memory during compilation";
        }
    }

    void Add(const TGraphiteMetricMapping& mapping) {
        TLabels labels;

        TString pattern = mapping.GetPattern();

        long maxIdx = -1;

        Y_ENSURE(RE2::FullMatch(pattern, *PATTERN_RE), "Pattern is illformated");

        for (const auto& labelMapping: mapping.GetLabelMappings()) {
            const auto& label = labelMapping.GetLabel();
            Y_ENSURE(RE2::FullMatch(label, *LABEL_RE), "Invalid label value " << label);

            const auto& value = labelMapping.GetTemplate();
            Y_ENSURE(RE2::FullMatch(value, *TEMPLATE_RE), "Invalid template " << value);
            maxIdx = Max(maxIdx, static_cast<long>(MaxPlaceholderIdx(value)));

            labels.Add(label, value);
        }

        auto wildCount = NormalizePattern(pattern);

        Y_ENSURE(maxIdx == -1 || (maxIdx > 0 && maxIdx <= static_cast<long>(wildCount)),
            "Placeholder reference is out of range");

        TMetricTemplate temp;
        temp.Pattern = std::move(pattern);
        temp.Labels = labels;
        temp.WildCount = wildCount;
        Map_.push_back(std::move(temp));

        /// Insert user-defined regexps into a matcher
        std::string errorMsg;
        i64 reIdx = AnyMatch_.Add({Map_.back().Pattern}, &errorMsg);

        if (-1 == reIdx) {
            const TString pattern = Map_.back().Pattern;
            Map_.pop_back();

            ythrow yexception() << "Failed to add a pattern " << pattern << " to a set: " << errorMsg;
        }
    }

    TMaybe<NMonitoring::TLabels> Convert(TStringBuf metric) const override {
        TVector<TStringBuf> substitutions;

        TVector<int> matchedIndices;
        if (!AnyMatch_.Match({static_cast<std::string>(metric)}, &matchedIndices)) {
            return Nothing();
        }

        // If a metric was matched by a several regexps, the first will be selected
        Sort(matchedIndices);
        const auto idx = matchedIndices[0];
        const auto& entry = Map_[idx];

        // values are set through a chain argPointers[i] -> args[i] -> captures[i]
        std::array<re2::StringPiece, WILD_LIMIT> captures;
        std::array<re2::RE2::Arg, WILD_LIMIT> args;
        std::array<re2::RE2::Arg*, WILD_LIMIT> argPointers;

        if (entry.WildCount) {
            for (auto i = 0u; i < entry.WildCount; ++i) {
                args[i] = &captures[i];
                argPointers[i] = &args[i];
            }

            bool isMatched = re2::RE2::FullMatchN(
                re2::StringPiece{metric.data(), metric.length()},
                entry.Pattern,
                argPointers.data(),
                entry.WildCount
            );
            Y_ENSURE(isMatched, "Failed to match all captures in a metric "
                                 << metric << " with a pattern " << entry.Pattern);

            substitutions.reserve(entry.WildCount);
            for (auto i = 0u; i < entry.WildCount; ++i) {
                Y_ENSURE(!captures[i].empty(), "Failed to extract " << i << "th capture value");
                substitutions.emplace_back(captures[i].begin(), captures[i].end());
            }
        }

        TLabels result;
        auto putSimple = [&] (const auto& label, auto idx) {
            if (idx == NO_REPLACE_IDX) {
                result.Add(label);
            } else if (idx == SINGLE_REPLACE_IDX) {
                TStringBuf v = label.Value();
                const auto idx =::FromString<ui32>(v.Skip(1));

                result.Add(label.Name(), substitutions[idx - 1]);
            }
        };

        for (const auto& label: entry.Labels) {
            if (re2::RE2::FullMatch(Sp(label.Value()), *SIMPLE_PART_RE)) {
                putSimple(label, NO_REPLACE_IDX);
            } else if (re2::RE2::FullMatch(Sp(label.Value()), *SIMPLE_REF_RE)) {
                putSimple(label, SINGLE_REPLACE_IDX);
            } else {
                auto labelValue = FillPlaceholders(label.Value(), substitutions);
                result.Add(label.Name(), labelValue);
            }
        }

        return result;
    }

private:
    struct TMetricTemplate {
        TString Pattern;
        NMonitoring::TLabels Labels;
        size_t WildCount {0};
    };

    TGraphiteConversionConfig::EMode Mode_;
    TVector<TMetricTemplate> Map_;
    re2::RE2::Set AnyMatch_;
};

ILabelConverterPtr CreateGraphiteLabelConverter(const TGraphiteConversionConfig& conf) {
    return ::MakeHolder<TGraphiteMetricMap>(conf);
}

class TGraphiteDecoder: public IDecoder {
public:
    TGraphiteDecoder(const TGraphiteConversionConfig& config)
        : Mode_{config.GetMode()}
        , LabelConverter_{MakeCaching({CreateGraphiteLabelConverter(config)})}
        , Blacklist_{config}
    {
    }

    void Decode(TStringBuf data, NMonitoring::IMetricConsumer& consumer) const override {
        // graphite plaintext protocol is plain space-separated 3-tuple:
        // <metric path> <metric value> <metric timestamp>

        consumer.OnStreamBegin();
        TStringBuf path;
        double value;
        ui64 ts;

        for (auto it: StringSplitter(data).Split('\n').SkipEmpty()) {
            TStringBuf line = it.Token();

            try {
                StringSplitter(line).Split(' ').SkipEmpty().CollectInto(&path, &value, &ts);
            } catch (...) {
                ythrow TFormatException() << CurrentExceptionMessage();
            }

            if (IsBlacklisted(path)) {
                continue;
            }
            auto labels = LabelConverter_->Convert(path);

            if (labels.Get()) {
                // nothing
            } else if (Mode_ == TGraphiteConversionConfig::IGNORE) {
                continue;
            } else if (Mode_ == TGraphiteConversionConfig::AS_IS) {
                labels = {{"sensor", path}};
            }

            consumer.OnMetricBegin(EMetricType::GAUGE);

            consumer.OnLabelsBegin();
            for (const auto& l: labels.GetRef()) {
                consumer.OnLabel(l.Name(), l.Value());
            }
            consumer.OnLabelsEnd();

            consumer.OnDouble(TInstant::Seconds(ts), value);

            consumer.OnMetricEnd();
        }

        consumer.OnStreamEnd();
    }

private:
    bool IsBlacklisted(TStringBuf name) const {
        return Blacklist_.Contains(name);
    }

private:
    TGraphiteConversionConfig::EMode Mode_;
    ILabelConverterPtr LabelConverter_;
    TGraphiteBlacklist Blacklist_;
};

IDecoderPtr CreateGraphiteDecoder(const TGraphiteConversionConfig& config) {
    return MakeHolder<TGraphiteDecoder>(config);
}

} // namespace NAgent
} // namespace NSolomon
